<?php

namespace App\Repositories\SoftwareLicense;

use App\Models\SoftwareLicense\SoftwareLicense;
use  App\Models\ZoomMember;
use App\Models\SoftwareLicense\SoftwareLicenseHistory;
use App\Models\SoftwareLicense\SoftwareLicenseUser;

class ZoomDataRepository
{

    protected $licenseName;
    protected $softwareLicenseModel;
    protected $softwareAssetMemberModel;
    protected $softwareLicenseUserModel;
    protected $licenseHistoryModel;
    
    /**
     * Constructor for ZoomDataRepository
     * 
     * @param SoftwareLicense $softwareLicenseModel
     * @param ZoomMember $softwareAssetMemberModel
     * @param SoftwareLicenseUser $softwareLicenseUserModel
     * @param SoftwareLicenseHistory $licenseHistoryModel
     */
    public function __construct(
        SoftwareLicense $softwareLicenseModel, ZoomMember $softwareAssetMemberModel, 
        SoftwareLicenseUser $softwareLicenseUserModel, SoftwareLicenseHistory $licenseHistoryModel
    ) 
    {
        $this->licenseName = 'Zoom';
        $this->softwareLicenseModel = $softwareLicenseModel;
        $this->softwareAssetMemberModel = $softwareAssetMemberModel;
        $this->softwareLicenseUserModel = $softwareLicenseUserModel;
        $this->licenseHistoryModel = $licenseHistoryModel;
    }

    /**
     * get license asset data
     * 
     * @return mixed
     */
    public function getLicenseData() 
    {
        return $this->softwareLicenseModel->where('name', $this->licenseName)->whereNull('deleted_at')->first();
    }

    /**
     * create license asset data
     * 
     * @param array $data
     * 
     * @return mixed
     */
    public function createLicenseData($data) {
        return $this->softwareLicenseModel->create($data);
    }

    /**
     * update license asset data
     * 
     * @param int $id
     * @param array $data
     * 
     * @return mixed
     */
    public function updateLicenseData($id, $data) 
    {
        return $this->softwareLicenseModel->findOrFail($id)->update($data);
    }

    /**
     * add history for the license actions
     */
    public function addLicenseHistory($data)
    {
        return $this->licenseHistoryModel->insertOrIgnore($data);
    }

    /**
     * clear existing data before fetching latest data
     * 
     * @return void
     */
    public function clearExistingDataBeforeFetching()
    {
        $this->softwareAssetMemberModel->truncate();
    }

    /**
     * It takes an array of members from software asset API,and add them to license's member table    
     * @param array members The array of members returned from the API
     * 
     * @return void
    */
    public function createMembers($members)
    { 
        $this->softwareAssetMemberModel->insert($members);       
    }

    /**
     * get member data of the software  assets
     * 
     * @return mixed
     */
    public function getMemberData()
    {
        return $this->softwareAssetMemberModel->whereNotNull('user_id')->get();
    }

    /**
     * get old users in the software license users whom are not valid now
     * 
     * @param int $softwareAssetId
     * 
     * @return array
     */
    public function getOldUsersToBeDeleted($softwareAssetId)
    {
        return $this->softwareLicenseUserModel->notZoomMembers($softwareAssetId)->pluck('user_id');
    }

    /**
     * delete old users in the software license users whom are not valid now
     * 
     * @param int $softwareAssetId
     * @param array $usersToBeDeleted
     * 
     * @return array
     */
    public function deleteOldUsersToBeDeleted($softwareAssetId, $usersToBeDeleted)
    {
        return $this->softwareLicenseUserModel
            ->where('software_license_id', $softwareAssetId)
            ->whereIn('user_id', $usersToBeDeleted)
            ->delete();
    }

    /**
     * update and create new users to the License
     * 
     * @param array $conditions
     * @param array $data
     * 
     * @return object
     */
    public function updateLicenseUsers($conditions, $data)
    {
        return  $this->softwareLicenseUserModel->updateOrCreate($conditions, $data);
    }

    /**
     * get owner details of the software license
     * 
     * @param string $ownerRole
     * 
     * @return mixed
     */
    public function getSoftwareLicenseOwner($ownerRole)
    {
        return $this->softwareAssetMemberModel->whereNotNull('user_id')->where('role', $ownerRole)->first();
    }

    /**
     * get active user count of the software license
     * 
     * @param string $ownerRole
     * 
     * @return mixed
     */
    public function getSoftwareLicenseActiveUserCount()
    {
        return $this->softwareAssetMemberModel->where('status','active')->count();
    }
}