<?php

namespace App\Repositories\SoftwareLicense;

use App\Models\ApiCredential;
use App\Models\SoftwareLicense\SoftwareLicense;
use App\Models\SoftwareLicense\PaymentMethod;
use App\Models\SoftwareLicense\SoftwareLicenseUser;
use App\Models\SoftwareLicense\SoftwareLicenseDocument;
use App\Models\SoftwareLicense\SoftwareLicenseHistory;
use App\Models\SoftwareLicense\SoftwareLicenseCategory;
use App\Models\SoftwareLicense\SoftwareLicenseKey;
use App\Models\SoftwareLicense\SoftwareLicenseManufacturer;
use App\Models\SoftwareLicense\SoftwareLicenseSubscription;
use App\Models\Vendor;

/**
 * Software License Repository
 */
class LicenseRepository
{

    protected $licenseModel;
    protected $paymentMethod;
    protected $licenseUserModel;
    protected $licenseDocumentModel;
    protected $licenseHistoryModel;
    protected $licenseCategoryModel;
    protected $softwareLicenseManufacturerModel;
    protected $licenseKeyModel;
    protected $licenseSubscriptionModel;

    /**
     * Constructor for Software License Repository
     */
    public function __construct(
        SoftwareLicense $licenseModel,
        PaymentMethod $paymentMethod,
        SoftwareLicenseUser $licenseUserModel,
        SoftwareLicenseDocument $licenseDocumentModel,
        SoftwareLicenseHistory $licenseHistoryModel,
        SoftwareLicenseCategory $licenseCategoryModel,
        SoftwareLicenseManufacturer $softwareLicenseManufacturerModel,
        SoftwareLicenseKey $licenseKeyModel,
        SoftwareLicenseSubscription $licenseSubscriptionModel,
        Vendor $vendorModel
    ) {
        $this->licenseModel = $licenseModel;
        $this->paymnentMethod = $paymentMethod;
        $this->licenseUserModel = $licenseUserModel;
        $this->licenseDocumentModel = $licenseDocumentModel;
        $this->licenseHistoryModel = $licenseHistoryModel;
        $this->licenseCategoryModel = $licenseCategoryModel;
        $this->softwareLicenseManufacturerModel = $softwareLicenseManufacturerModel;
        $this->licenseKeyModel = $licenseKeyModel;
        $this->licenseSubscriptionModel = $licenseSubscriptionModel;
        $this->vendorModel = $vendorModel;
    }

    /**
     * get all Software License payment methods
     */
    public function getAllPaymentMethods()
    {
        return $this->paymnentMethod->where('status', 1);
    }

    /**
     * Create Software License
     * 
     * @param array $data
     * 
     * @return object
     */
    public function createLicense($data)
    {
        return $this->licenseModel->create($data);
    }

    /**
     * List all the Software Licenses
     * 
     */
    public function getAllLicenses()
    {
        return $this->licenseModel->with(['category', 'softwareLicenseManufacturer', 'licenseUser', 'vendor']);
    }

    /**
     * get Software License data
     * 
     * @param int $id
     * 
     * @return object
     */
    public function getLicense(int $id)
    {
        return $this->licenseModel->with(['category', 'softwareLicenseManufacturer', 'history.user', 'paymentmethod', 'licenseKey'])->find($id);
    }

    /**
     * update Software License
     * 
     * 
     */
    public function updateLicense($data)
    {
        return $this->licenseModel->where('id', $data['id'])->update($data);
    }

    /**
     * List Software License users
     */
    public function getLicenseUsers($licenseId)
    {
        return $this->licenseUserModel->with(['user.userType', 'licenseKey'])->where('software_license_id', $licenseId);
    }

    /**
     * get License User
     */
    public function getLicenseUser($userId, $licenseId)
    {
        return $this->licenseUserModel->where('user_id', $userId)->where('software_license_id', $licenseId);
    }

    /**
     * Create Software License User
     * 
     * @param array $data
     * @return object
     */
    public function createLicenseUser($data)
    {
        return $this->licenseUserModel->create($data);
    }

    /**
     * Remove User from Software License
     *
     * @param int $id
     * 
     * @return bool
     */
    public function removeUserFromLicense(int $id)
    {
        return $this->licenseUserModel->where('id', $id)->delete();
    }


    /**
     * Save documents of the license 
     */
    public function saveDocuments($data)
    {
        return $this->licenseDocumentModel->create($data);
    }

    /**
     * Get document of the license 
     */
    public function getDocument($id)
    {
        return $this->licenseDocumentModel->find($id);
    }

    /**
     * Delete document of the license 
     */
    public function deleteDocument($id)
    {
        return $this->licenseDocumentModel->where('id', $id)->delete($id);
    }

    /**
     * add history for the license actions
     */
    public function addLicenseHistory($data)
    {
        return $this->licenseHistoryModel->create($data);
    }

    /**
     * get license user data with id
     */
    public function getUserFromLicenseData($id)
    {
        return $this->licenseUserModel->find($id);
    }

    /**
     * filters with exact match with the serch value 
     */
    public function filterByExactMatch($query, $filterKey, $filterValue)
    {
        if (is_array($filterValue)) {
            return  $query->whereIn($filterKey, $filterValue);
        } else {
            return $query->where($filterKey, $filterValue);
        }

        return $query;
    }

    /**
     * filters with text match with the serch value
     */
    public function filterByTextMatch($query, $filterKey, $filterValue)
    {
        return $query->where($filterKey, 'like', '%' . $filterValue . '%');
    }

    /**
     * filters with date range match with the serch value 
     */
    public function filterByDateRangeMatch($query, $filterKey, $filterValue)
    {
        if (strpos($filterKey, '_from') !== false) {
            $query = $query->whereDate(str_replace('_from', '', $filterKey), '>=', convert_to_db_date($filterValue));
        }

        if (strpos($filterKey, '_to') !== false) {
            $query = $query->whereDate(str_replace('_to', '', $filterKey), '<=', convert_to_db_date($filterValue));
        }

        return $query;
    }

    /**
     * filters with date range match with the serch value 
     */
    public function filterByCountMatch($query, $filterKey, $filterValue)
    {
        return $query->having($filterKey, $filterValue);
    }

    /**
     * List Software License documents
     * 
     * @param int $licenseId
     */
    public function getLicenseDocuments(int $licenseId)
    {
        return $this->licenseDocumentModel->where('software_license_id', $licenseId);
    }

    /**
     * get License Document with id
     * 
     * @param int $id
     * 
     * @return object
     */
    public function getLicenseDocument(int $id)
    {
        return $this->licenseDocumentModel->find($id);
    }

    /**
     * Delete Software License data
     * 
     * @param int $id
     * 
     * @return object
     */
    public function deleteLicense(int $id)
    {
        return $this->licenseModel->where('id', $id)->delete();
    }

    /**
     * Delete all users from Software License
     * 
     * @param int $licenseId
     * 
     * @return object
     */
    public function deleteLicenseUsers(int $licenseId)
    {
        return $this->licenseUserModel->where('software_license_id', $licenseId)->delete();
    }

    /**
     * Delete all documents from Software License
     * 
     * @param int $licenseId
     * 
     * @return object
     */
    public function deleteLicenseDocuments(int $licenseId)
    {
        return $this->licenseDocumentModel->where('software_license_id', $licenseId)->delete();
    }

    /**
     * Delete all histories of Software License
     * 
     * @param int $licenseId
     * 
     * @return object
     */
    public function deleteLicenseHistories(int $licenseId)
    {
        return $this->licenseHistoryModel->where('software_license_id', $licenseId)->delete();
    }

    /**
     * Get software license category id with the name
     * 
     * @param array $item
     * @param string $dbField
     * @param string $fileHeader
     * 
     * @return int
     */
    public function getSoftwareLicenseCategoryId($item, $dbField, $fileHeader)
    {
        if (!empty($item[$fileHeader])) {
            $softwareLicenseCategory = $this->licenseCategoryModel->where('name', $item[$fileHeader])->first();

            return $softwareLicenseCategory ? $softwareLicenseCategory->id : null;
        }

        return null;
    }

    /**
     * Get software license Manufacturer id with the name
     * 
     * @param array $item
     * @param string $dbField
     * @param string $fileHeader
     * 
     * @return int
     */
    public function getSoftwareLicenseManufacturerId($item, $dbField, $fileHeader)
    {
        if (!empty($item[$fileHeader])) {
            $softwareLicenseManufacturer = $this->softwareLicenseManufacturerModel->where('name', $item[$fileHeader])->first();

            return $softwareLicenseManufacturer ? $softwareLicenseManufacturer->id : null;
        }

        return null;
    }

    /**
     * Get software license payment method id with the name
     * 
     * @param array $item
     * @param string $dbField
     * @param string $fileHeader
     * 
     * @return int
     */
    public function getSoftwareLicensePaymentmethodId($item, $dbField, $fileHeader)
    {
        if (!empty($item[$fileHeader])) {
            $softwareLicensePaymentMethod = $this->paymnentMethod->where('name', $item[$fileHeader])->first();

            return $softwareLicensePaymentMethod ? $softwareLicensePaymentMethod->id : null;
        }

        return null;
    }

    /**
     * Get software license subscription length id with the name
     * 
     * @param array $item
     * @param string $dbField
     * @param string $fileHeader
     * 
     * @return int
     */
    public function getSoftwareLicenseSubscriptionId($item, $dbField, $fileHeader)
    {
        if (!empty($item[$fileHeader])) {
            $softwareLicenseSubscription = $this->licenseSubscriptionModel->where('name', $item[$fileHeader])->first();

            return $softwareLicenseSubscription ? $softwareLicenseSubscription->id : null;
        }

        return null;
    }


    /**
     * Get vendor id with the name
     * 
     * @param array $item
     * @param string $dbField
     * @param string $fileHeader
     * 
     * @return int
     */
    public function getVendorId($item, $dbField, $fileHeader)
    {
        if (!empty($item[$fileHeader])) {
            $vendor = $this->vendorModel->where('name', $item[$fileHeader])->first();

            return $vendor ? $vendor->id : null;
        }

        return null;
    }


    /**
     * List Software License keys
     * 
     * @param int $licenseId
     */
    public function getAllLicenseKeys(int $licenseId)
    {
        return $this->licenseKeyModel->with('licenseUsers')->where('software_license_id', $licenseId);
    }



    /**
     * get License Key with id
     * 
     * @param int $id
     * 
     * @return object
     */
    public function getLicenseKey(int $id)
    {
        return $this->licenseKeyModel->find($id);
    }

    /**
     * Delete all keys from Software License
     * 
     * @param int $licenseId
     * 
     * @return object
     */
    public function deleteLicenseKeys(int $licenseId)
    {
        $this->licenseKeyModel->where('software_license_id', $licenseId)->delete();
    }

    /**
     * Update all keys from Software License
     * 
     * @param int $licenseId
     * @param array $updateData
     * 
     * @return object
     */
    public function updateLicenseKeys(int $licenseId, $updateData = null)
    {
        return $this->licenseKeyModel->where('software_license_id', $licenseId)->update($updateData);
    }


    /**
     * Delete key of the license 
     */
    public function deleteKey($id)
    {
        return $this->licenseKeyModel->where('id', $id)->delete($id);
    }

    /**
     * Get userwise software assets for search asset page total value
     * @param mixed $user=null
     * 
     * @return [type]
     */
    public function assetUserSearched($user = null)
    {
        return SoftwareLicenseUser::with(['license'])->where('user_id', $user);
    }

    /**
     * The function turns off the integration status of an API credential based on the name of a
     * software license.
     * @param id The parameter "id" is the identifier of a software license record in the database.
     */
    public function turnOffIntegrationStatus($licenseId)
    {
        $softwareAssetName = SoftwareLicense::select('name')->find($licenseId)->name ?? '';
        ApiCredential::where('slug', strtolower($softwareAssetName))->update(['integration_status' => 0]);
    }

    /**
     * This function deletes all users from integration associated with a given software license .
     * 
     * @param licenseId The ID of a software license 
     */
    public function deleteIntegrationUsers($licenseId)
    {
        $softwareAsset  = SoftwareLicense::find($licenseId);
        $namespace      = "\App\Models\\";
        $nonTeqtivityUserClassName = $namespace . ($softwareAsset->name ? $softwareAsset->name . 'Member' : '');

        if (!class_exists($nonTeqtivityUserClassName)) {

            return false;
        }

        $class = new $nonTeqtivityUserClassName();

        return $class->truncate();
    }
}
