<?php

namespace App\Repositories\Settings\History;

use App\Models\AssetHistory;
use App\Models\PurgeAssetHistory;
use App\Models\SoftwareLicense\SoftwareLicenseHistory;

class AssetsHistoryRepository
{
    /**
     * Constructor for the class.
     *
     * @param AssetHistory $assetHistoryModel The asset history model.
     * @param SoftwareLicenseHistory $softwareLicenseHistoryModel The software license history model.
     */
    public function __construct(
        protected AssetHistory $assetHistoryModel,
        protected SoftwareLicenseHistory $softwareLicenseHistoryModel
    ) {}

    /**
     * Retrieve asset history data.
     *
     * @return mixed
     */
    public function getAssetHistoryData()
    {
        return $this->assetHistoryModel::with(
            'user:id,first_name,last_name',
            'newUser:id,first_name,last_name',
            'oldUser:id,first_name,last_name',
        );
    }

    /**
     * Get the system history data.
     *
     * @return mixed
     */
    public function getSystemHistoryData()
    {
        return $this->getAssetHistoryData()->where('description', '!=', '');
    }

    /**
     * Get the user history data.
     *
     * @return mixed
     */
    public function getUserHistoryData()
    {
        return $this->getAssetHistoryData()
            ->whereNotIn('action', ['linked', 'moved', 'created'])
            ->where(function ($query) {
                $query->whereNotNull('new_user_id')->orWhereNotNull('old_user_id');
            })
            ->where('description', '!=', '');
    }

    /**
     * Get the location history data.
     *
     * @return mixed
     */
    public function getLocationHistoryData()
    {
        return $this->getAssetHistoryData()
            ->locationHistoryAll()
            ->where('description', '!=', '');
    }

    /**
     * Filter the query with a specific location ID.
     *
     * @param mixed $query The query to filter
     * @param mixed $location_id The ID of the location to filter by
     * @return mixed The modified query
     */
    public function filterWithLocation($query, $locationId)
    {
        return $query->when($locationId, function ($query) use ($locationId) {
            $query->where(function ($query) use ($locationId) {
                if (is_array($locationId) === TRUE) {
                    return $query->whereIn('new_location_id', $locationId)->orWhereIn('old_location_id', $locationId);
                }
                return $query->where('new_location_id', $locationId)->orWhere('old_location_id', $locationId);
            });
        });
    }

    /**
     * Filter the query with the specified user ID.
     *
     * @param datatype $query The query to filter
     * @param datatype $userId The ID of the user to filter with
     * @return mixed
     */
    public function filterWithUser($query, $userId)
    {
        return $query->when($userId, function ($query) use ($userId) {
            $query->where(function ($query) use ($userId) {
                $query->orWhere('new_user_id', $userId)->orWhere('old_user_id', $userId);
            });
        });
    }

    /**
     * Retrieves the purged assets history data with the associated user and manufacturer.
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function purgedAssetsHistoryData()
    {
        return PurgeAssetHistory::with('user', 'makeAndModel.manufacturer');
    }

    /**
     * Retrieves the software asset history data.
     *
     * This function queries the `SoftwareLicenseHistory` table and retrieves the software asset history data.
     * It includes the associated user data and filters out any records with a null description.
     *
     * @return \Illuminate\Database\Eloquent\Builder The query builder instance.
     */
    public function getSoftwareAssetHistoryData()
    {
        return $this->softwareLicenseHistoryModel->with('user')->whereNotNull('description');
    }
}
