<?php

namespace App\Repositories\Security\Crowdstrike;

use App\Models\Asset;
use App\Models\AssetStatus;
use App\Models\AssetType;
use App\Models\AssetVulnerability;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class CrowdstrikeReportRepository
{

    protected $assetStatusModel;
    protected $assetTypeModel;
    protected $assetModel;
    protected $assetVulnerabilityModel;
    
    /**
     * Constructor for CrowdstrikeReportRepository
     * 
     */
    public function __construct(
        AssetStatus $assetStatusModel, AssetType $assetTypeModel, 
        Asset $assetModel, AssetVulnerability $assetVulnerabilityModel
    ) 
    {
        $this->assetStatusModel = $assetStatusModel;
        $this->assetTypeModel = $assetTypeModel;
        $this->assetModel = $assetModel;
        $this->assetVulnerabilityModel = $assetVulnerabilityModel;
    }

    /**
     * Retrieves the sensor detection data.
     *
     * @return object The assets that have the status 'assigned', 'loaned', or 'installed' and the type 'laptop' or 'desktop'.
     */
    public function sensorDetectionData()
	{
		$statusIds = $this->assetStatusModel->whereIn('slug', ['assigned', 'loaned', 'installed'])->get()->pluck('id');
        $typeIds = $this->assetTypeModel->whereIn('slug', ['laptop', 'desktop', 'computer'])->get()->pluck('id');
        $assets = $this->assetModel->whereIn('asset_status_id', $statusIds)
                    ->whereIn('asset_type_id', $typeIds);

		return $assets;
	}

    /**
     * Retrieves the vulnerability data for sensors.
     *
     * @return Illuminate\Database\Eloquent\Collection
     */
    public function sensorVulnerabilityData()
    {
        $statusIds = $this->assetStatusModel->whereIn('slug', ['assigned', 'loaned', 'installed'])->get()->pluck('id');
        $typeIds = $this->assetTypeModel->whereIn('slug', ['laptop', 'desktop', 'computer'])->get()->pluck('id');

        $assets = $this->assetVulnerabilityModel->with('asset')
            ->whereHas('asset', function ($query) use ($statusIds, $typeIds) {
                $query->whereIn('asset_status_id', $statusIds)
                    ->whereIn('asset_type_id', $typeIds);
            });

		return $assets;
    }

    /**
     * Generate the function comment for the given function body in a markdown code block with the correct language syntax.
     *
     * @return QueryBuilder
     */
    public function activeCveData()
    {
        $activeCves = AssetVulnerability::select('asset_vulnerabilities.*')
            ->selectRaw('COUNT(DISTINCT id) as vulnerability_count')
            ->groupBy('cve_id');

        return $activeCves;
    }


    /**
     * Filter with the asset's fields 
     *
     * @param mixed $query The query to filter.
     * @param string $field The field to filter on.
     * @param mixed $value The value to filter with.
     * 
     * @return mixed The filtered query.
     */
    public function filterWithAssetFields($query, $field, $value)
	{
		if ($value == "") {
			return $query;
		}

		if (is_array($value)) {
			return $query->whereIn($field, $value);
		}

		return $query->where($field, $value);
	}

    /**
     * Fiter with asset's user fields
     *
     * @param $query DB Query
     * @param $field Filter filed name
     * @param $value Filter field value
     *
     * @return DB Query
     */
    public function filterWithAssetUserFields($query, $field, $value)
    {
        if ($value == "") {
            return $query;
        }

        return $query->whereHas('user', function ($query) use ($field, $value) {
            if (is_array($value)) {
                $query->whereIn($field, $value);
            } else {
                $query->where($field, $value);
            }
        });
    }

    /**
     * Filters the asset's custom fields 
     *
     * @param mixed $query The query to filter.
     * @param string $field The field to filter on.
     * @param mixed $value The value to filter with.
     * 
     * @return mixed The filtered query.
     */
    public function filterWithAssetCustomFields($query, $field, $value)
	{
		if ($value == "") {
			return $query;
		}

        if ($field == 'crowdstrike_id') {

            if ($value == 'Yes') {
              return $query->whereNotNull($field);
            } 
            
            return $query->whereNull($field);
            
        }

        return $query;
	}

    /**
     * Filters the vulnerabilities with vulnerability's fields
     *
     * @param mixed $query The query to filter.
     * @param string $field The field to filter on.
     * @param mixed $value The value to filter by.
     * @return mixed The filtered query.
     */
    public function filterWithVulnerabilityFields($query, $field, $value)
	{
		if ($value == "") {
			return $query;
		}

        if ($field == 'days_open') {
            return $query->whereDate('opened_at', '=', Carbon::now()->subDays(round($value)));
        }



        if ($field == 'vulnerability_count') {
            
            if ($value == 0) {
                return $query->having('vulnerability_count', '>', $value);
            } else {
                return $query->having('vulnerability_count', $value);
            }
        }

		if (is_array($value)) {
			return $query->whereIn($field, $value);
		}

		return $query->where($field, $value);
	}

    /**
     * Filters the vulnerabilities date fields with the range
     *
     * @param $query The query to filter.
     * @param $filterKey The key to filter.
     * @param $filterValue The value to filter.
     * @return The filtered query.
     */
    public function filterVulnerabilityDateRangeMatch($query, $filterKey, $filterValue)
    {
        if (strpos($filterKey, '_from') !== false) {
            $query = $query->whereDate(str_replace('_from', '', $filterKey), '>=', convert_to_db_date($filterValue));
        }

        if (strpos($filterKey, '_to') !== false) {
            $query = $query->whereDate(str_replace('_to', '', $filterKey), '<=', convert_to_db_date($filterValue));
        }

        return $query;
    }

    /**
     * Fiter with asset and it's fields on which the vulnerabilities were discovered
     *
     * @param $query DB Query
     * @param $field Filter filed name
     * @param $value Filter field value
     *
     * @return DB Query
     */
public function filterWithVulnerabilitysAssetFields($query, $field, $value)
    {
        if ($value == "") {
            return $query;
        }

        return $query->whereHas('asset', function ($query) use ($field, $value) {
                
            if (is_array($value)) {
                $query->whereIn($field, $value);
            } else {
                $query->where($field, $value);
            }
        });
    }

    /**
     * Fiter with user's fields of the asset on which the vulnerabilities were discovered
     *
     * @param $query DB Query
     * @param $field Filter filed name
     * @param $value Filter field value
     *
     * @return DB Query
     */
    public function filterWithVulnerabilitysAssetsUserFields($query, $field, $value)
    {
        if ($value == "") {
            return $query;
        }

        return $query->whereHas('asset.user', function ($query) use ($field, $value) {
            $query->where($field, $value);
        });
    }

    /**
     * Retrieves a single CVE data by its ID.
     *
     * @param int $id The ID of the CVE data.
     * 
     * @return AssetVulnerability The retrieved CVE data.
     */
    public function getSingleCveData($id)
    {
        return AssetVulnerability::find($id);
    }

    
    /**
     * Retrieves the assets associated with a specific CVE.
     *
     * @param int $cveId The ID of the CVE.
     * @return \Illuminate\Database\Eloquent\Builder The query builder instance.
     */
    public function getcveAssets($cveId)
    {
        return AssetVulnerability::with('asset')
            ->where('cve_id', $cveId);
    }

    /**
     * Retrieves the asset types along with their count for the given CVE asset IDs.
     *
     * @param array $cveAssetsIds The array of CVE asset IDs.
     * 
     * @return \Illuminate\Support\Collection The collection of asset types along with their count.
     */
    public function getAssetTypesWithCount($cveAssetsIds)
    {
        return AssetType::select('asset_types.name')
            ->addSelect(DB::raw('COUNT(assets.id) as asset_count'))
            ->join('assets', 'assets.asset_type_id', '=', 'asset_types.id')
            ->whereIn('assets.id', $cveAssetsIds)
            ->groupBy('asset_types.id')
            ->get();
    }

    /**
     * Returns distinct vulnerability fields based on the specified type and search term.
     *
     * @param string $type The type of vulnerability field to filter by.
     * @param string $search The search term to filter by.
     * 
     * @return Collection The distinct vulnerability fields.
     */
    public function getFilterVulnerabilityFields($search, $type)
    {
        return $this->assetVulnerabilityModel->distinct()->select("$type")->where("$type", 'like', $search . '%')->get();
    }

    /**
     * Retrieves the count of active vulnerabilities for a given week.
     *
     * @param datatype $firstDay The first day of the week.
     * @param datatype $lastDay The last day of the week.
     * 
     * @return int The count of active vulnerabilities.
     */
    public function getActiveVulnerabilitiesCountByWeek($firstDay, $lastDay)
    {
        return $this->assetVulnerabilityModel::whereBetween('opened_at', [
            $firstDay, $lastDay
            ])
            ->count();
    }
}