<?php

namespace App\Repositories\Reports;

use App\Models\Asset;
use App\Models\AssetStatus;
use App\Models\AssetType;
use Carbon\Carbon;
use DB;

/**
 * Repository Class for Use Tax Report
 */
class UseTaxReportRepository
{
    /**
     * Taking the asset status details for `Use Tax` Report
     *
     * @param null
     * @return App\Models\AssetStatus
     */
    public function getAssetStatus()
    {
        return AssetStatus::reportStatus()->orderBy('name', 'ASC')->get();
    }

    /**
     * Making asset query with latest asset history.
     *
     * @param null
     * @return DB Query String
     */

    public function getAssets()
    {
        return Asset::with(['assetType:id,name', 'makeAndModel:id,name,manufacturer_id', 'makeAndModel.manufacturer:id,name', 'assetStatus:id,name', 'latestAssetHistory', 'user:id,first_name,last_name,status,city,department_id,hire_date', 'user.department:id,name', 'location:id,room_name']);
    }

    /**
     * Calculating days between latest histories
     *
     * @param $latestHistory   App\Models\AssetHistory
     * @param $AssetCreatedDate    App\Models\AssetHistory
     *
     * @return int
     */
    public function daysAtCurrentLocation($latestHistory, $assetCreatedDate)
    {
        $endDate = Carbon::now();
        if ($latestHistory) {
            $startDate = (validateDate(optional($latestHistory)->CreatedDate)) ? date('Y-m-d H:i:s', strtotime((string) optional($latestHistory)->CreatedDate)) : false;
        } else {
            $startDate = (validateDate($assetCreatedDate)) ? date('Y-m-d H:i:s', strtotime((string) $assetCreatedDate)) : false;
        }
        return getDaysDifference($startDate, $endDate);
    }
    /*
    * Search with asset fields
    */
    public function searchAssetFields($assets, $field, $value)
    {
        if ($value) {
            $assets = is_array($value) ? $assets->whereIn($field, $value) : $assets->where($field, $value);
        }
        return $assets;
    }

    /**
     * make query to filter laptop or monitor assets
     */
    public function filterWithAssetType($assets)
    {
        $assetTypeIds = AssetType::getBySlugList(['laptop', 'monitor', 'computer'])->pluck('id')->toArray();

        if ($assetTypeIds) {
            $assets = $assets->whereIn('asset_type_id', $assetTypeIds);
        }

        return $assets;
    }

    /**
     * Filters the given assets based on their asset status.
     */
    public function filterWithAssetStatus($assets)
    {
        $assetStatusIds = AssetStatus::whereIn('slug', ['stolen_lost', 'end_of_life', 'end_of_life_data_wiped', 'rma_returned'])
            ->orWhere('slug', 'LIKE', 'eol%')
            ->get()->modelKeys();

        if ($assetStatusIds) {
            $assets = $assets->whereNotIn('asset_status_id', $assetStatusIds);
        }

        return $assets;
    }
}
