<?php

namespace App\Repositories\Reports;

use App\Models\Asset;
use App\Models\AssetStatus;
use Illuminate\Support\Facades\DB;

class UnrepairableRepository
{

    protected $assetModel;
    protected $damagedStatusId;
    protected $unrepairedStatuses;

    /**
     * Constructs a new instance of the class.
     *
     * @param Asset $assetModel The asset model.
     * @param Manufacturer $manufacturerModel The manufacturer model.
     */
    public function __construct(Asset $assetModel)
    {
        $this->assetModel = $assetModel;
        $this->damagedStatusId = $this->getDamagedAssetStatus();
        $this->unrepairedStatuses = $this->getUnrepairedAssetStatus();
    }



    /**
     * get query for unrepairable assets
     * @return [type]
     */
    public function getUnrepairableAssets()
    {
        $assets = $this->assetModel->with('location', 'user', 'assetType', 'makeAndModel.manufacturer', 'technicalSpec', 'assetStatus', 'assetHistory')
            ->whereHas('assetHistory', function ($query) {
                $query->where('old_asset_status_id', $this->damagedStatusId)
                        ->whereIn('new_asset_status_id', $this->unrepairedStatuses);
            });
        return $assets;
    }

    /**
     * Pre defined damaged status
     * @return integer
     */
    public function getDamagedAssetStatus()
    {
        return optional(AssetStatus::where('slug', 'damaged')->first())->id;
    }

    /**
     * Pre defiend set of all asset status that are considered as unrepaired
     * @return []
     */
    public function getUnrepairedAssetStatus()
    {
        return AssetStatus::where('slug', ['end_of_life'])->get()->modelKeys();
    }

    /**
     * get query for all damaged assets
     *
     * @return assets
     */
    public function getTotalDamagedAssets()
    {
        $assets = $this->assetModel->with('location', 'user', 'assetType', 'makeAndModel.manufacturer', 'assetStatus', 'assetHistory')
                    ->whereHas('assetHistory', function ($query)  {
                        $query->where('new_asset_status_id', $this->damagedStatusId);
                    });

        return $assets;
    }

    /**
     * query for getting all damaged assets under warranty
     *
     * @param date $dateFrom
     * 
     * @return assets
     */
    public function getAssetsUnderWarranty($dateFrom)
    {
        $assets = $this->assetModel->with('location', 'user', 'assetType', 'makeAndModel.manufacturer', 'assetStatus', 'assetHistory')
            ->whereHas('assetHistory', function ($query) {
                $query->where('old_asset_status_id', $this->damagedStatusId);
            })
            ->when($dateFrom, function ($query) use ($dateFrom) {
                $query->whereDate('warranty_end_date', '>=', convert_to_db_date($dateFrom));
            });

        return $assets;
    }

    /**
     * get total days that taken by an asset from damaged to unrepaired status
     *
     * @param date $dateFrom
     * @param date $dateTo
     *
     * @return count
     */
    public function retrieveDaysTakenToUnrepairAssets($dateFrom, $dateTo)
    {
        $unrepairedAssets = DB::table('asset_histories as ah1')
            ->join('asset_histories as ah2', 'ah1.asset_id', '=', 'ah2.asset_id')
            ->whereIn('ah1.new_asset_status_id', $this->unrepairedStatuses)
            ->where('ah2.new_asset_status_id', $this->damagedStatusId)
            ->when($dateFrom, function ($query) use ($dateFrom) {
                return $query->where('ah1.created_at', '>=', convert_to_db_date($dateFrom));
            })
            ->when($dateTo, function ($query) use ($dateTo) {
                return $query->where('ah1.created_at', '<=', convert_to_db_date($dateTo));
            });

        if (request()->ajax()) {
            $unrepairedAssets = $this->filterUnrepaired($unrepairedAssets);
        }

        $daysTaken = $unrepairedAssets
            ->sum(DB::raw('DATEDIFF(ah2.created_at, ah1.created_at)'));

        return $daysTaken;
    }

    /**
     * query to retrieve the created date of the latest asset histories entry where the status was changed to damaged
     *
     * @param $asset
     *
     * @return date
     */
    public function getDateDamaged($asset)
    {
        return $asset->latestAssetHistoryBasedOnStatus($this->damagedStatusId);
    }

    /**
     * query to retrieve the created date of the latest asset histories entry where the status was changed to un repaired statuses
     *
     * @param $asset
     *
     * @return date
     */
    public function getDateUpdated($asset)
    {
        return $asset->latestAssetHistoryBasedOnStatus($this->unrepairedStatuses);
    }

    /**
     * query to filter the unrepaired assets based on the form data
     *
     * @param $asset
     *
     * @return assets
     */
    public function filterUnrepaired($unrepairedAssets)
    {
        $requestedData = request()->form ?? request()->all();
        $unrepairedAssets->join('assets', 'ah2.asset_id', 'assets.id');

        if (array_key_exists('asset_type', $requestedData) && $requestedData['asset_type']) {
            $unrepairedAssets->where('asset_type_id', $requestedData['asset_type']);
        }

        if (array_key_exists('make_and_model', $requestedData) && $requestedData['make_and_model']) {
            $unrepairedAssets->where('make_and_model_id', $requestedData['make_and_model']);
        }

        if (array_key_exists('status', $requestedData) && $requestedData['status']) {
            $unrepairedAssets->where('new_asset_status_id', $requestedData['status']);
        }
        
        if (array_key_exists('location', $requestedData) && $requestedData['location']) {
            $unrepairedAssets->where('location_id', $requestedData['location']);
        }

        return $unrepairedAssets;
    }

}
