<?php

namespace App\Repositories\Reports;

use App\Models\Asset;
use App\Models\TermAssetsCollectionDetails;

class TermsCollectionReportRepository
{
    protected $assetModel;
    protected $termAssetsCollectionDetailsModel;

    /**
     * Constructs a new instance of the class.
     *
     * @param Asset $AssetModel The Asset model.
     * @param TermAssetsCollectionDetails $termAssetsCollectionDetailsModel The TermAssetsCollectionDetails model.
     */
    public function __construct(Asset $assetModel, TermAssetsCollectionDetails $termAssetsCollectionDetailsModel)
    {
        $this->assetModel = $assetModel;
        $this->termAssetsCollectionDetailsModel = $termAssetsCollectionDetailsModel;
    }
    /**
     * makes query to get assets received from terminated users
     */
    public function getCollectedAssets()
    {
        return $this->assetModel::with('assetStatus:id,name','assetType:id,name', 'makeAndModel:id,name', 'user:id,email,first_name,last_name', 'latestAssetHistory:id,asset_id,updated_at')
            ->with(['latestAssetHistory' => function ($query) {
                $query->select('id', 'asset_id', 'ticket_no')->has('assignedUserAsset');
            }])
            ->select('assets.id','assets.user_id','assets.make_and_model_id','assets.asset_status_id','assets.asset_type_id', 'assets.serial_no', 'assets.asset_tag','assets.created_at','assets.updated_at')
            ->whereHas('User', function ($query) {
                $query->where('status', '!=', 1);
            })
            ->receivedAsset();
    }

    /**
     * makes query to get assets which are not collected from terminated users
     */
    public function getUnCollectedAssets()
    {
        return $this->assetModel::with('assetStatus:id,name','assetType:id,name', 'makeAndModel:id,name', 'user:id,email,first_name,last_name', 'latestAssetHistory:id,asset_id,updated_at')
            ->with(['latestAssetHistory' => function ($query) {
                $query->select('id', 'asset_id', 'ticket_no')->has('assignedUserAsset');
            }])
            ->select('assets.id','assets.user_id','assets.make_and_model_id','assets.asset_status_id','assets.asset_type_id', 'assets.serial_no', 'assets.asset_tag','assets.created_at','assets.updated_at')
            ->whereHas('User', function ($query) {
                $query->where('status', '!=', 1);
            })
            ->termAsset();
    }

    /**
     * filters query by user terminated_date
     */
    public function filterAssetsByTerminatedDateRange($assets, $dateFrom, $dateTo)
    {
        $assets->whereHas('User', function ($userQuery) use ($dateFrom, $dateTo) {
            $userQuery->when(!empty($dateFrom), function ($subQuery) use ($dateFrom) {
                $subQuery->where('terminated_date', '>=', $dateFrom);
            })->when(!empty($dateTo), function ($subQuery) use ($dateTo) {
                $subQuery->where('terminated_date', '<=', $dateTo);
            });
        });

        return $assets;
    }

    /**
     * Retrieves the data by executing the query.
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function getData()
    {
        return TermAssetsCollectionDetails::whereHas('asset', function($query) {
            return $query->termAsset();
        });
    }

    /**
     * Saves the collection details for a given set of assets.
     *
     * @param array $assets The assets to save.
     * @param int $year The year of the collection details.
     * @param int $month The month of the collection details.
     * @param string $status The status of the collection details.
     * @param string $reportType The report type of the collection details.
     * @return void
     */
    public function saveCollectionDetails($assets, $year, $month, $status, $reportType)
    {
        $this->termAssetsCollectionDetailsModel::where([
            'year' => $year,
            'month' => $month,
            'collection_status' => $status,
            'report_type' => $reportType
        ])->delete();

        foreach ($assets as $asset) {
            $this->termAssetsCollectionDetailsModel::firstOrCreate([
                'report_type' => $reportType,
                'asset_id' => $asset->id,
                'user_id' => $asset->user_id,
                'year' => $year,
                'month' => $month,
                'collection_status' => $status
            ]);
        }
    }
}