<?php

namespace App\Repositories\Reports;

use App\Models\Asset;
use App\Models\AssetHistory;
use App\Models\AssetStatus;
use App\User;

class RetrievalStatisticsWidgetRepository
{
    protected $assetModel;
    protected $userModel;
    protected $assetStatusModel;
    protected $assetHistoryModel;

    /**
     * @param Asset $assetModel
     * @param User $userModel
     * @param AssetStatus $assetStatusModel
     * @param AssetHistory $assetHistoryModel
     */
    public function __construct(Asset $assetModel, User $userModel, AssetStatus $assetStatusModel, AssetHistory $assetHistoryModel)
    {
        $this->assetModel = $assetModel;
        $this->userModel = $userModel;
        $this->assetStatusModel = $assetStatusModel;
        $this->assetHistoryModel = $assetHistoryModel;
    }
    /**
     * Retrieves the number of terminations within a given date range.
     *
     * @param string $startDate The start date of the date range.
     * @param string $endDate The end date of the date range.
     * @return int The count of terminations within the date range.
     */
    public function getTerminationsCount($startDate, $endDate)
    {
        return $this->userModel::whereNotNull('terminated_date')
            ->when($startDate != '', function ($query) use ($startDate) {
                $query->where('terminated_date', '>=', convert_to_db_date($startDate));
            })->when($endDate != '', function ($query) use ($endDate) {
                $query->where('terminated_date', '<=', convert_to_db_date($endDate));
            })->count();
    }

    /**
     * Retrieves the count of assigned assets within a specified date range.
     *
     * @param string $startDate The start date of the range.
     * @param string $endDate The end date of the range.
     * @return int The count of assigned assets.
     */
    public function getAssetsAssignedCount($startDate = '', $endDate = '')
    {
        $assetsFilterStatusIds = $this->assetStatusModel::whereIn('slug', config('reports.retrieval_statistics.coming_back_statuses'))->pluck('id')->toArray();

        return $this->assetModel::whereIn('asset_status_id', $assetsFilterStatusIds)
            ->whereHas('user', function ($query) use($startDate, $endDate) {
                $query->whereNotNull('terminated_date');
                $query->when($startDate != '', function ($query) use ($startDate) {
                    $query->where('terminated_date', '>=', convert_to_db_date($startDate));
                });
                $query->when($endDate != '', function ($query) use ($endDate) {
                    $query->where('terminated_date', '>=', convert_to_db_date($endDate));
                });
            })->count();
    }

    /**
     * Retrieves the count of assets that have been returned within a specified date range.
     *
     * @param string $startDate The start date of the range to filter the assets.
     * @param string $endDate The end date of the range to filter the assets.
     *
     * @return int The count of assets that have been returned within the specified date range.
     */
    public function getAssetsReturnedCount($startDate, $endDate)
    {
        $assetsComingBackStatusIds = $this->assetStatusModel::whereIn('slug', config('reports.retrieval_statistics.coming_back_statuses'))->pluck('id')->toArray();
        $assetsUnrecoverableStatusIds = $this->assetStatusModel::whereIn('slug', config('reports.retrieval_statistics.coming_back_statuses'))->pluck('id')->toArray();

        return $this->assetHistoryModel::whereHas('asset')
            ->whereNotIn('new_asset_status_id', $assetsComingBackStatusIds)
            ->whereNotIn('new_asset_status_id', $assetsUnrecoverableStatusIds)
            ->whereHas('oldUser', function ($query) use($startDate, $endDate) {
                $query->whereNotNull('terminated_date');
                $query->when($startDate != '', function ($query) use ($startDate) {
                    $query->where('terminated_date', '>=', convert_to_db_date($startDate));
                });
                $query->when($endDate != '', function ($query) use ($endDate) {
                    $query->where('terminated_date', '>=', convert_to_db_date($endDate));
                });
            })->count();
    }
    /**
     * Retrieves the count of assets with unrecoverable status within a given date range.
     *
     * @param string $startDate The start date of the range.
     * @param string $endDate The end date of the range.
     * @return int The count of assets with unrecoverable status.
     */
    public function getAssetsUnrecoverableCount($startDate, $endDate)
    {
        $assetsUnrecoverableStatusIds = $this->assetStatusModel::whereIn('slug', config('reports.retrieval_statistics.coming_back_statuses'))->pluck('id')->toArray();

        return $this->assetHistoryModel::whereHas('asset')
            ->whereIn('new_asset_status_id', $assetsUnrecoverableStatusIds)
            ->whereHas('oldUser', function ($query) use($startDate, $endDate) {
                $query->whereNotNull('terminated_date');
                $query->when($startDate != '', function ($query) use ($startDate) {
                    $query->where('terminated_date', '>=', convert_to_db_date($startDate));
                });
                $query->when($endDate != '', function ($query) use ($endDate) {
                    $query->where('terminated_date', '>=', convert_to_db_date($endDate));
                });
            })->count();
    }

    /**
     * Retrieves the count of assets that have been coming back within a specified date range.
     * @param string $startDate
     * @param string $endDate
     *
     */
    public function getComingBackAssets($startDate = '', $endDate = '')
    {
        return $this->assetHistoryModel::with('asset', 'asset.assetStatus', 'oldUser')
            ->whereHas('asset')
            ->whereHas('newUser', function ($query) use($startDate, $endDate) {
                $query->whereNotNull('terminated_date');
                $query->when($startDate != '', function ($query) use ($startDate) {
                    $query->where('terminated_date', '>=', convert_to_db_date($startDate));
                });
                $query->when($endDate != '', function ($query) use ($endDate) {
                    $query->where('terminated_date', '>=', convert_to_db_date($endDate));
                });
            });
    }

    /**
     * Retrieves the next status of an asset after it has been assigned to a user.
     *
     * @param int $assetId The ID of the asset.
     * @param int $userId The ID of the user.
     * @return AssetHistory|null The next status of the asset after it has been assigned, or null if not found.
     */
    public function getNextStatusAfterAssigned($assetId, $userId)
    {
        return $this->assetHistoryModel::where('old_user_id', $userId)
            ->where('asset_id', $assetId)
            ->whereNotIn('new_asset_status_id', config('reports.retrieval_statistics.unrecoverable_statuses'))
            ->latest()
            ->first();
    }
}