<?php

namespace App\Repositories\Reports;

use App\Models\AssetHistory;
use App\Models\AssetStatus;

class RetrievalStatisticsRepository
{
    protected $assetHistoryModel;
    protected $assetStatusModel;

    public function __construct(AssetHistory $assetHistoryModel, AssetStatus $assetStatusModel)
    {
        $this->assetHistoryModel = $assetHistoryModel;
        $this->assetStatusModel = $assetStatusModel;
    }
    /**
     * Get the data.
     *
     * @return AssetHistory
     */
    public function getData()
    {
        return $this->assetHistoryModel::with('Asset', 'newUser', 'newUser.department')->whereHas('asset');
    }

    /**
     * Retrieves the report list data based on the provided input data.
     *
     * @param array $inputData The input data used to filter the report list.
     * @return \Illuminate\Database\Eloquent\Builder The query builder instance.
     */
    public function getReportListData($inputData)
    {
        return $this->assetHistoryModel::with('asset', 'asset.assetStatus')
            ->when($inputData['type'] == 1, function ($query) use ($inputData) {
                $query->with('newUser', 'newUser.department');
            })
            ->when($inputData['type'] == 2, function ($query) use ($inputData) {
                $query->with('oldUser', 'oldUser.department');
            })->whereHas('asset');
    }

    /**
     * Retrieves the asset returned history based on the asset ID, user ID, and termination date.
     *
     * @param int $assetId The ID of the asset.
     * @param int $userId The ID of the user.
     * @param string $terminationDate The termination date.
     * @return AssetHistory|null The asset history object or null if not found.
     */
    public function getAssetReturnedHistory($assetId, $userId, $terminationDate)
    {
        return $this->assetHistoryModel::where('old_user_id', $userId)
            ->where('asset_id', $assetId)
            ->whereDate('created_at', '>=', convert_to_db_datetime($terminationDate))
            ->latest()
            ->first();
    }

    /**
     * Retrieves assets that are coming back based on the provided query parameters.
     *
     * @param mixed $query the query object used to retrieve the assets
     * @param mixed $fromDate the starting date to filter the assets by
     * @param mixed $toDate the ending date to filter the assets by
     * @return mixed the modified query object with the applied filters
     */
    public function assetsComingBack($query, $fromDate, $toDate)
    {
        $assetComingBackStatusIds = $this->assetStatusModel::whereIn('slug', config('reports.retrieval_statistics.coming_back_statuses'))->pluck('id')->toArray();

        $query->whereHas('newUser', function ($userQuery) use ($fromDate, $toDate) {
            $userQuery->whereNotNull('terminated_date');
            $userQuery->whereDate('terminated_date', '>=', $fromDate);
            $userQuery->whereDate('terminated_date', '<=', $toDate);
        });
        $query->whereIn('new_asset_status_id', $assetComingBackStatusIds);

        return $query;
    }

    /**
     * Retrieve assets that are reported as lost or stolen within a specific date range.
     *
     * @param mixed $query The query builder object to retrieve the assets.
     * @param mixed $fromDate The starting date of the range.
     * @param mixed $toDate The ending date of the range.
     * @return mixed The modified query builder object.
     */
    public function assetsLostStolen($query, $fromDate, $toDate)
    {
        $assetLostStolenStatusIds = $this->assetStatusModel::whereIn('slug', config('reports.retrieval_statistics.unrecoverable_statuses'))->pluck('id')->toArray();

        $query->whereHas('oldUser', function ($userQuery) use ($fromDate, $toDate) {
            $userQuery->whereNotNull('terminated_date');
            $userQuery->whereDate('terminated_date', '>=', $fromDate);
            $userQuery->whereDate('terminated_date', '<=', $toDate);
        });
        $query->whereIn('new_asset_status_id', $assetLostStolenStatusIds);

        return $query;
    }

    /**
     * Filters the query for assets by type.
     *
     * @param mixed $query The query to filter.
     * @param int $type The type of assets to filter by.
     * @return mixed The filtered query.
     */
    public function filterQueryForAssetsByType($query, $type)
    {
        $assetsFilterStatusIds = [];

        if ($type == 1) {
            $assetsFilterStatusIds = $this->assetStatusModel::whereIn('slug', config('reports.retrieval_statistics.coming_back_statuses'))->pluck('id')->toArray();
        } else if ($type == 2) {
            $assetsFilterStatusIds = $this->assetStatusModel::whereIn('slug', config('reports.retrieval_statistics.unrecoverable_statuses'))->pluck('id')->toArray();
        }
        $query->whereIn('new_asset_status_id', $assetsFilterStatusIds);

        return $query;
    }

    /**
     * Retrieves the next status of an asset after it has been assigned to a user.
     *
     * @param int $assetId The ID of the asset.
     * @param int $userId The ID of the user.
     * @return AssetHistory|null The next status of the asset after it has been assigned, or null if not found.
     */
    public function getNextStatusAfterAssigned($assetId, $userId)
    {
        return $this->assetHistoryModel::where('old_user_id', $userId)
            ->where('asset_id', $assetId)
            ->whereNotIn('new_asset_status_id', config('reports.retrieval_statistics.unrecoverable_statuses'))
            ->latest()
            ->first();
    }
}