<?php

namespace App\Repositories\Reports;

use App\Models\AssetHistory;
use App\Models\AssetStatus;
use Illuminate\Support\Facades\DB;

class RepairAndRecoveryRepository
{
    
    protected $assetHistoryModel;

    /**
     * Constructs a new instance of the class.
     *
     * @param AssetHistory $assetHistoryModel The asset history model.
     */
    public function __construct(AssetHistory $assetHistoryModel)
    {
        $this->assetHistoryModel = $assetHistoryModel;
        $this->damagedStatusId = $this->getDamagedAssetStatus();
        $this->repairedStatuses = $this->getRepairedAssetStatus();
    }
    
    /**
     * get query for all repaired assets
     * @return [type]
     */
    public function getRepairAndRecoveryAssets()
    {
        $assets = $this->assetHistoryModel
            ->with(
            'asset:id,asset_tag,technical_spec_id,serial_no,asset_type_id,make_and_model_id,asset_status_id,user_id,location_id,created_at,asset_original_value,warranty_end_date',
            'asset.latestAssetHistory:id,asset_id,user_id,updated_at',
            'asset.latestAssetHistory.user:id,first_name,last_name',
            'asset.assetType:id,name', 'asset.assetStatus:id,name',
            'asset.technicalSpec:id,original_value'
            )
            ->where('old_asset_status_id', $this->damagedStatusId)
            ->whereIn('new_asset_status_id', $this->repairedStatuses);
        
        return $assets;
    }

    /**
     * get query for all damaged(repaired pending) assets
     * @return [type]
     */
    public function getRepairAndRecoveryPendingAssets()
    {
        $assets = $this->assetHistoryModel->with('asset')
                    ->where('new_asset_status_id', $this->damagedStatusId)
                    ->where(function ($query) {
                        $query->whereRaw('created_at = (
                            SELECT MAX(created_at)
                            FROM asset_histories AS ah
                            WHERE ah.asset_id = asset_histories.asset_id
                        )');
                    });

        return $assets;
    }

    /**
     * Pre defined damaged status 
     * @return [type]
     */
    public function getDamagedAssetStatus()
    {
        return optional(AssetStatus::where('slug', 'damaged')->first())->id;
    }

    /**
     * Pre defined set of all asset status that are considered as repaired
     * @return [type]
     */
    public function getRepairedAssetStatus()
    {  
        return AssetStatus::whereIn('slug', config('reports.repair_and_recovery.include_statuses'))->get()->modelKeys();
    }

    /**
     * Retrieves the total damaged assets.
     *
     * @return assets
     */
    public function getTotalDamagedAssets()
    {
        $assets = $this->assetHistoryModel->with('asset')
                    ->where('new_asset_status_id', $this->damagedStatusId)
                    ->where(function ($query) {
                        $query->whereRaw('created_at = (
                            SELECT MAX(created_at)
                            FROM asset_histories AS ah
                            WHERE ah.asset_id = asset_histories.asset_id
                        )');
                    });

        return $assets;
    }

    /**
     * Retrieves the total repaired asset count.
     *
     * @return count
     */
    public function getRepairedAssetCount($dateFrom, $dateTo)
    {
        $assets = $this->assetHistoryModel->with('asset')
                    ->where('old_asset_status_id', $this->damagedStatusId)
                    ->whereIn('new_asset_status_id', $this->repairedStatuses);
        if ($dateFrom) {
            $assets->whereDate('created_at', '>=', convert_to_db_date($dateFrom));
        }

        if ($dateTo) {
            $assets->whereDate('created_at', '<=', convert_to_db_date($dateTo));
        }

        return $assets->count();
    }

    /**
     * Retrieves the total repaired pending assets.
     *
     * @return assets
     */
    public function getRepairedPendingAssets()
    {
        $assets = $this->assetHistoryModel->with('asset')
                    ->where('new_asset_status_id', $this->damagedStatusId)
                    ->where(function ($query) {
                        $query->whereRaw('created_at = (
                            SELECT MAX(created_at)
                            FROM asset_histories AS ah
                            WHERE ah.asset_id = asset_histories.asset_id
                        )');
                    });

        return $assets;
    }


    /**
     * get query for sum of total days that taken by an asset from damaged to repaired status 
     * 
     * @param date $dateFrom
     * @param date $dateTo
     * 
     * @return count
     */
    public function retrieveDaysTakenRepairAssets($dateFrom, $dateTo)
    {
        $latestRepaired = DB::table('asset_histories as ah1')
            ->whereIn('ah1.new_asset_status_id', $this->repairedStatuses)
            ->where('ah1.old_asset_status_id', $this->damagedStatusId);
        
            
        $latestDamaged = DB::table('asset_histories as ah2')
            ->joinSub($latestRepaired, 'ah1', function ($join) {
                $join->on('ah2.asset_id', '=', 'ah1.asset_id'); 
            })
            ->where('ah2.new_asset_status_id', $this->damagedStatusId)
            ->whereBetween('ah1.created_at', [convert_to_db_date($dateFrom), convert_to_db_date($dateTo)]);
        
        if (request()->ajax()) {
            $latestDamaged = $this->filterDamaged($latestDamaged);
        }

        $daysTaken = $latestDamaged->sum(DB::raw('DATEDIFF(ah1.created_at, ah2.created_at)'));

        return $daysTaken;
    }

    /**
     * query to retrieve the created date of the latest asset histories entry where the status was changed to damaged
     * 
     * @param $asset
     * 
     * @return date
     */
    public function getDateDamaged($asset)
    {
        return $asset->latestAssetHistoryBasedOnStatus($this->damagedStatusId);
    }

    /**
     * query to retrieve the created date of the latest asset histories entry where the status was changed to repaired statuses
     * 
     * @param $asset
     * 
     * @return date
     */
    public function getDateRepaired($asset)
    {
        return $asset->latestAssetHistoryBasedOnStatus($this->repairedStatuses);
    }

    /**
     * query to filter the unrepaired assets based on the form data
     *
     * @param $asset
     *
     * @return assets
     */
    public function filterDamaged($latestDamaged)
    {
        $requestedData = request()->form ?? request()->all();
        $latestDamaged->join('assets', 'ah2.asset_id', 'assets.id');

        if (array_key_exists('asset_type', $requestedData) && $requestedData['asset_type']) {
            $latestDamaged->where('asset_type_id', $requestedData['asset_type']);
        }

        if (array_key_exists('make_and_model', $requestedData) && $requestedData['make_and_model']) {
            $latestDamaged->where('make_and_model_id', $requestedData['make_and_model']);
        }

        if (array_key_exists('status', $requestedData) && $requestedData['status']) {
            $latestDamaged->where('new_asset_status_id', $requestedData['status']);
        }
        
        if (array_key_exists('location', $requestedData) && $requestedData['location']) {
            $latestDamaged->where('location_id', $requestedData['location']);
        }

        return $latestDamaged;
    }
}
