<?php

namespace App\Repositories\Reports;

use App\Models\Asset;
use App\Models\AssetType;
use App\Models\MakeAndModel;
use App\Models\Manufacturer;
use App\Models\TechnicalSpecs;
use Facades\App\Repositories\HardwareStandards;
use Illuminate\Support\Facades\DB;

class DepreciationValueRepository
{
    /**
     * Retrieves the data from the database.
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function getData()
    {
        return Asset::with(['assetStatus:id,name', 'makeAndModel:id,name,manufacturer_id', 'makeAndModel.manufacturer:id,name', 'technicalSpec:id,details', 'user:id,first_name,last_name','location:id,room_name']);
    }

    /**
     * Retrieves all asset types.
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getAssetTypes()
    {
        return AssetType::orderBy('slug')->get();
    }

    /**
     * Filters the make and models based on the provided input.
     *
     * @param mixed $makeAndModel The make and model(s) to filter by. Can be either an array or a single value.
     * @return array The filtered make and model(s).
     */
    public function filterMakeAndModels($makeAndModel)
    {
        return MakeAndModel::when(is_array($makeAndModel), function ($query) use ($makeAndModel) {
                $query->whereIn('id', $makeAndModel);
            })->when(!is_array($makeAndModel), function ($query) use ($makeAndModel) {
                $query->where('id', $makeAndModel);
            })
            ->get()
            ->toArray();
    }

    /**
     * Retrieves the hardware standards based on the given asset type.
     *
     * @param mixed $assetType The asset type to retrieve hardware standards for.
     * @return mixed The hardware standards for the given asset type.
     */
    public function getHardwareStandardsByAssetTypes($assetType)
    {
        return HardwareStandards::getHardwareStandardsByAssetTypes([request('asset_type')]);
    }

    /**
     * Find the name of an asset type.
     *
     * @param int $assetType The ID of the asset type.
     * @return string The name of the asset type.
     */
    public function findAssetTypeName($assetType)
    {
        return AssetType::find($assetType)->name;
    }

    /**
     * Retrieves the list of hardware standards.
     *
     * @return array The list of hardware standards.
     */
    public function getHardwareStandardsList()
    {
        return HardwareStandards::getHardwareStandardsList();
    }

    /**
     * Retrieves the technical specifications from the database.
     *
     * @return TechnicalSpecs[] Returns an array of TechnicalSpecs objects ordered by details.
     */
    public function getTechnicalSpecs()
    {
        return TechnicalSpecs::orderBy('details')->get();
    }

    /**
     * Filters the technical specifications.
     *
     * @param mixed $technicalSpecs The technical specifications to filter.
     * @return Some_Return_Value The filtered technical specifications.
     */
    public function filterTechnicalSpecs($technicalSpecs)
    {
        return TechnicalSpecs::GetById($technicalSpecs)->get();
    }

    /**
     * Retrieves the asset original values for a given technical specification.
     *
     * @param mixed $technicalSpecs The technical specifications object.
     * @return mixed Returns an array of asset original values.
     */
    public function getAssetsForChart($technicalSpecs)
    {
        return Asset::select(DB::raw('count(*) as count'), DB::raw('SUM(CASE WHEN assets.asset_original_value IS NULL THEN technical_specs.original_value ELSE assets.asset_original_value END) AS total_original_value'))
            ->leftJoin('technical_specs', 'assets.technical_spec_id', '=', 'technical_specs.id')
            ->where('technical_spec_id', $technicalSpecs->id)
            ->first();
    }

    /**
     * Filters the manufactures based on the given criteria.
     *
     * @param mixed $manufacture The manufacture criteria to filter by.
     * @return array The filtered manufactures as an array.
     */
    public function filterManufactures($manufacture)
    {
        return Manufacturer::when(is_array($manufacture), function ($query) use ($manufacture) {
                $query->whereIn('id', $manufacture);
            })->when(!is_array($manufacture), function ($query) use ($manufacture) {
                $query->where('id', $manufacture);
            })
            ->get()
            ->toArray();
    }

    public function getTechnicalSpecsQuery()
    {
        return TechnicalSpecs::query();
    }
}