<?php

namespace App\Repositories\Reports;

use App\Models\AssetHistory;
use App\Models\AssetType;
use App\Models\Department;
use App\Models\Location;
use App\Models\MakeAndModel;

class DamagedTrendsRepository
{
    /**
     * Constructor for the class.
     *
     * @param AssetHistory $assetHistoryModel The asset history model.
     * @param AssetType $assetTypeModel The asset type model.
     * @param MakeAndModel $makeAndModelModel The make and model model.
     * @param Location $locationModel The location model.
     * @param Department $departmentModel The department model.
     */
    public function __construct(
        protected AssetHistory $assetHistoryModel,
        protected AssetType $assetTypeModel,
        protected MakeAndModel $makeAndModelModel,
        protected Location $locationModel,
        protected Department $departmentModel
    ) {
        $this->assetHistoryModel = $assetHistoryModel;
        $this->assetTypeModel = $assetTypeModel;
        $this->makeAndModelModel = $makeAndModelModel;
        $this->locationModel = $locationModel;
        $this->departmentModel = $departmentModel;
    }

    /**
     * Retrieve data from the asset history model with a condition on the new value being 'Damaged'.
     *
     * @return mixed
     */
    public function getData()
    {
        return $this->assetHistoryModel::with('asset')->where('new_value', 'Damaged');
    }

    /**
     * Get the filters for the data options.
     *
     * @return array
     */
    public function getFilters()
    {
        $filterDataOptions['assetTypes'] = $this->assetTypeModel::valid()->getWithoutAssetTypes(config('reports.damaged_trends.exclude_asset_types'))->orderBy('name', 'ASC')->get();
        $filterDataOptions['makeAndModels'] = $this->makeAndModelModel::valid()->getWithoutAssetTypes(config('reports.damaged_trends.exclude_asset_types'))->orderBy('name')->get();
        $filterDataOptions['locations'] = $this->locationModel::orderBy('room_name')->get();
        $filterDataOptions['departments'] = $this->departmentModel::orderBy('name')->get();

        return $filterDataOptions;
    }

    /**
     * Get the asset ID for the laptop.
     *
     * @return int
     */
    public function getLaptopAssetId()
    {
        return  $this->assetTypeModel::whereIn('slug', config('reports.damaged_trends.laptop_asset_types'))->first()->id;
    }

    /**
     * Retrieve the make and models of laptops based on the given asset types.
     *
     * @param array $assetTypes The array of asset types to filter the make and models by.
     * @return Collection The retrieved make and models of laptops.
     */
    public function getLaptopMakeAndModels($assetTypes = [])
    {
        return $this->makeAndModelModel::getWithAssetTypes($assetTypes)->get();
    }

    /**
     * Retrieves make and model data based on their IDs.
     *
     * @param array $makeAndModelIds The array of make and model IDs to retrieve data for.
     * @return Illuminate\Support\Collection The collection of make and model data.
     */
    public function getMakeAndModelsByIds($makeAndModelIds)
    {
        return $this->makeAndModelModel::whereIn('id', $makeAndModelIds)->get();
    }
}