<?php

namespace App\Repositories;

use App\Models\Asset;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

/**
 * Repositor Class for Use Tax Report
 */
class ReportGeneratorRepository
{
    /**
     * Query for report generator
     */
    public function reportGeneratorQuery()
    {
        $assets = Asset::select('assets.*')
            ->leftJoin('technical_specs', 'assets.technical_spec_id', '=', 'technical_specs.id')
            ->with('location', 'user', 'assetType', 'carrier', 'makeAndModel.manufacturer', 'technicalSpec', 'assetStatus', 'parentAsset', 'childrenAsset', 'latestAssetHistory');

        return $assets;
    }

    /**
     * Taking Filters for report generator
     * @param Http\Request
     * @return Array
     */
    public function getInputData()
    {
        $inputData = [];
        $inputData['asset_age_value'] = isset(request('form')['asset_age_value']) ? request('form')['asset_age_value'] : '';
        $inputData['asset_age_unit'] = isset(request('form')['asset_age_unit']) ? request('form')['asset_age_unit'] : '';
        $inputData['depreciated_value_from'] = isset(request('form')['depreciated_value_from']) ? request('form')['depreciated_value_from'] : '';
        $inputData['depreciated_value_to'] = isset(request('form')['depreciated_value_to']) ? request('form')['depreciated_value_to'] : '';
        foreach (config('asset-fields.select') as $key => $fields) {
            if (isset(request('form')[$key])) {
                $inputData[$key] = request('form')[$key];
            }
        }

        foreach (config('asset-fields.boolean') as $key => $fields) {
            if (isset(request('form')[$key])) {
                $inputData[$key] = request('form')[$key];
            }
        }

        foreach (config('asset-fields.dates_from') as $key => $fields) {
            if (isset(request('form')[$fields])) {
                $inputData[$fields] = request('form')[$fields];
            }
        }
        foreach (config('asset-fields.dates_to') as $key => $fields) {
            if (isset(request('form')[$fields])) {
                $inputData[$fields] = request('form')[$fields];
            }
        }

        foreach (config('asset-fields.radioButton') as $key => $fields) {
            if (isset(request('form')[$key])) {
                $inputData[$key] = request('form')[$key];
            }
        }

        if (isset(request('form')['show_linked'])) {
            $showLinked = request('form')['show_linked'];
            $inputData['show_linked'] = $showLinked;
        }

        if (isset(request('form')['show_individual'])) {
            $showIndividual = request('form')['show_individual'];
            $inputData['show_individual'] = $showIndividual;
        }


        return $inputData;
    }

    /**
     * Taking the filters for Report generator Export
     *
     * @return Array
     */
    public function getExportInputData()
    {
        $inputData = [];
        $inputAssetAgeValue = request('asset_age_value');
        $inputAssetAgeUnit = request('asset_age_unit');
        $inputData['asset_age_value'] = isset($inputAssetAgeValue) ? $inputAssetAgeValue : '';
        $inputData['asset_age_unit'] = isset($inputAssetAgeUnit) ? $inputAssetAgeUnit : '';
        foreach (config('asset-fields.select') as $key => $fields) {
            $keyExist = request($key);
            if (isset($keyExist)) {
                $inputData[$key] = request($key) ?? '';
            }
        }

        foreach (config('asset-fields.boolean') as $key => $fields) {
            $keyExist = request($key);
            if (isset($keyExist)) {
                $inputData[$key] = request($key) ?? '';
            }
        }

        foreach (config('asset-fields.dates_from') as $key => $fields) {
            $keyExist = request($fields);
            if (isset($keyExist)) {
                $inputData[$fields] = request($fields) ?? '';
            }
        }

        foreach (config('asset-fields.dates_to') as $key => $fields) {
            $keyExist = request($fields);
            if (isset($keyExist)) {
                $inputData[$fields] = request($fields) ?? '';
            }
        }

        foreach (config('asset-fields.radioButton') as $key => $fields) {
            $keyExist = request($key);
            if (isset($keyExist)) {
                $inputData[$key] = request($key) ?? '';
            }
        }

        $inputShowLinked = request('show_linked');
        $inputShowIndividual = request('show_individual');
        $inputData['show_linked'] = isset($inputShowLinked) ? $inputShowLinked : '';
        $inputData['show_individual'] = isset($inputShowIndividual) ? $inputShowIndividual : '';


        return $inputData;
    }

    public function calculateAssetAgeDate($assetAge, $ageUnit = '')
    {
        if ($ageUnit == 'day') {
            $date = Carbon::now()->subDays($assetAge);
        }

        if ($ageUnit == 'month') {
            $date = Carbon::now()->subMonths($assetAge);
        }

        if ($ageUnit == 'year') {
            $date = Carbon::now()->subYears($assetAge);
        }

        return $date;
    }

    /**
     * Default Filter
     *
     * @param Query $assets
     * @param $field  Filter field
     * @param Array $value   Filter input data
     */
    public function filter($assets, $field, $value)
    {
        if ($value) {

            if ($value == 'null') {
                return $assets->where(function ($assets) use ($field) {
                    return $assets->whereNull($field)
                        ->orWhere($field, 0);
                });
            }

            if ($value == 'po_null') {
                return $assets->where(function ($assets) use ($field) {
                    return $assets->whereNull($field)
                        ->orWhere('po_id', '');
                });
            }

            if (is_array($value)) {
                $assets = $assets->whereIn('assets.'.$field, $value);
            } else {
                $assets = $assets->where('assets.'.$field, $value);
            }
        }

        return $assets;
    }

    /**
     * Filter with User fields
     *
     * @param Query $assets
     * @param $field  Filer field
     * @param Array $value   Filter input data
     */
    public function filterWithUserFields($assets, $field, $value)
    {
        if ($value) {
            $assets = $assets->whereHas('user', function ($query) use ($field, $value) {
                if (is_array($value)) {
                    $query->whereIn($field, $value);
                } else {
                    $query->where($field, $value);
                }
            });
        }
        return $assets;
    }

    /**
     * Filter with Lcoation fields
     *
     * @param Query $assets
     * @param $field  Filer field
     * @param Array $value   Filter input data
     */
    public function filterWithLocationFields($assets, $field, $value)
    {
        if ($value) {
            $assets = $assets->whereHas('location', function ($query) use ($field, $value) {
                if (is_array($value)) {
                    $query->whereIn($field, $value);
                } else {
                    $query->where($field, $value);
                }
            });
        }

        return $assets;
    }

    /**
     * Filter with makeAndModel fields
     *
     * @param Query $assets
     * @param $field  Filer field
     * @param Array $value   Filter input data
     */
    public function filterWithMakeAndModelFields($assets, $field, $value)
    {
        if ($value) {
            $assets = $assets->whereHas('makeAndModel', function ($query) use ($field, $value) {
                if (is_array($value)) {
                    $query->whereIn($field, $value);
                } else {
                    $query->where($field, $value);
                }
            });
        }
        return $assets;
    }

    /**
     * Filter with Location name
     *
     * @param Query $assets
     * @param $field  Filer field
     * @param Array $value   Fitler input data
     */
    public function filterWithLocationName($assets, $field, $value)
    {
        if ($value) {

            $assets = $assets->whereHas('location', function ($query) use ($field, $value) {
                $query->where($field, 'like', '%' . $value . '%')
                    ->orWhere('address', 'like', '%' . $value . '%')
                    ->orWhere('city', 'like', '%' . $value . '%');
            });
        }

        return $assets;
    }

    /**
     * Append asset history join query.
     *
     * @param datatype $assets description
     *
     * @return object
     */
    public function appendAssetHistoryJoinQuery($assets)
    {
        return $assets->join('asset_histories', 'assets.id', 'asset_histories.asset_id')
                ->where('asset_histories.id', function ($query) {
                    $query->select('id')
                        ->from('asset_histories')
                        ->whereColumn('asset_id', 'assets.id')
                        ->latest()
                        ->limit(1);
                });
    }

    /**
     * Filter assets with a specific value.
     *
     * @param mixed $assets The assets to filter.
     * @param mixed $value The value to filter by.
     * @return Collection
     */
    public function filterWithAssetValue($assets, $value)
    {
        return $assets->when(is_array($value), function ($query) use ($value) {
                $query->whereIn(DB::raw('COALESCE(assets.asset_original_value, technical_specs.original_value)'), $value);
            })
            ->when(!is_array($value), function ($query) use ($value) {
                $query->where(DB::raw('COALESCE(assets.asset_original_value, technical_specs.original_value)'), '=', $value);
            });
    }

    /**
     * Filter assets with depreciated value.
     *
     * @param mixed $assets
     * @param mixed $value
     * @return mixed
     */
    public function filterWithDepreciatedValue($assets, $value)
    {
        return $assets->when(is_array($value), function ($query) use ($value) {
                $query->whereIn(DB::raw('ROUND(CASE
                    WHEN (FLOOR((DATEDIFF(NOW(), assets.created_at)) / 365)) > 0 THEN
                        ROUND((COALESCE(assets.asset_original_value, technical_specs.original_value)) -
                        (((COALESCE(assets.asset_original_value, technical_specs.original_value)) * 20) / 100) *
                        (FLOOR((DATEDIFF(NOW(), assets.created_at)) / 365)), 2)
                    ELSE
                        (COALESCE(assets.asset_original_value, technical_specs.original_value))
                    END)'), $value);
            })
            ->when(!is_array($value), function ($query) use ($value) {
                $query->where(DB::raw('ROUND(CASE
                    WHEN (FLOOR((DATEDIFF(NOW(), assets.created_at)) / 365)) > 0 THEN
                        ROUND((COALESCE(assets.asset_original_value, technical_specs.original_value)) -
                        (((CASE WHEN assets.asset_original_value IS NULL THEN technical_specs.original_value ELSE assets.asset_original_value END) * 20) / 100) *
                        (FLOOR((DATEDIFF(NOW(), assets.created_at)) / 365)), 2)
                    ELSE
                        (COALESCE(assets.asset_original_value, technical_specs.original_value))
                    END)'), '=', $value);
            });
    }
}
