<?php

namespace App\Repositories\OsCompliance;

use App\Models\AirWatch;
use App\Models\AssetType;
use App\Models\OperatingSystemComplianceSettings;
use App\Repositories\DiscoveryTools\WorkspaceOneRepository;
use Illuminate\Support\Str;

class WorkspaceOneComplianceRepository extends AbstractOsComplianceRepository
{
    protected $workspaceOneRepository;

    public function __construct()
    {
        $this->workspaceOneRepository = new WorkspaceOneRepository;
    }
    /**
     * Making query for Workspace ONE devices with non compliant os version or platform
     * @param OperatingSystemComplianceSettings $settings
     */
    public function getDevices()
    {
        $assets =  AirWatch::select('air_watches.id', 'air_watches.serial_no', 'air_watches.model', 'air_watches.platform', 'air_watches.os_version', 'air_watches.asset_id', 'air_watches.last_seen', 'air_watches.user_id', 'air_watches.user_name', 'air_watches.email')
            ->with(
                'asset:id,serial_no,asset_type_id,make_and_model_id,technical_spec_id,asset_status_id,user_id,location_id',
                'asset.assetType:id,name',
                'asset.makeAndModel:id,name',
                'asset.technicalSpec:id,details',
                'asset.assetStatus:id,name',
                'asset.user:id,first_name,last_name',
                'asset.location:id,room_name'
            )
            ->join('assets', 'assets.serial_no', '=', 'air_watches.serial_no')
            ->whereNotIn('platform', ['Windows', 'WinRT', 'ChromeOS']);
        return $assets;
    }

    /**
     * Making query for android devices
     * @param Query $assets
     */
    public function filterAndroidDevices($assets)
    {
        return $assets->where('platform', '=', 'Android');
    }

    /**
     * Making query to filter apple mobile phones and tablets
     * @param Query $assets
     */
    public function filterMobileDevices($assets)
    {
        $assetTypeIds = AssetType::whereIn('slug', ['mobile_phone', 'tablet'])->get()->modelKeys();
        if ($assetTypeIds) {
            $assets->where(function ($query) use ($assetTypeIds) {
                $query->whereIn('assets.asset_type_id', $assetTypeIds)
                    ->orWhere('model', 'LIKE', 'iPhone%')
                    ->orWhere('model', 'LIKE', 'iPad%');
            })
                ->where('platform', '<>', 'Android');
        }

        return $assets;
    }

    /**
     * Making the query for Workspace ONE input filters
     */
    public function filterAppleDevices($assets, $inputData)
    {
        $assets = $assets->RelationData($inputData['mdm_serial_no'], 'air_watches.serial_no');
        $assets = $assets->RelationData($inputData['model'], 'air_watches.model');
        $assets = $assets->RelationData($inputData['user'], 'air_watches.email');
        $assets = $this->workspaceOneRepository->filterWithLastSeen($assets, $inputData['checkinFrom'], $inputData['checkinTo']);
        $assets = $assets->RelationData($inputData['platform'], 'air_watches.platform');
        $assets = $assets->RelationData($inputData['os_version'], 'air_watches.os_version');
        $assets = $this->filterAssets($assets, $inputData);

        return $assets;
    }

    /**
     * fetch Workspace ONE filter values for report
     * @param string $type Workspace ONE attribute name
     */
    public function getFilterData($type)
    {
        return AirWatch::select("$type")->whereNotIn('platform', ['Windows', 'WinRT', 'ChromeOS'])->groupBy($type)->get()->filter();
    }
}
