<?php

namespace App\Repositories\OsCompliance;

use App\Models\AssetType;
use App\Models\JamfComputer;
use App\Models\OperatingSystemComplianceSettings;
use App\Repositories\DiscoveryTools\JamfRepository;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class JamfComplianceRepository extends AbstractOsComplianceRepository
{
    protected $jamfRepository;

    public function __construct()
    {
        $this->jamfRepository = new JamfRepository;
    }

    /**
     * Making query for Jamf devices with non compliant os version or platform
     */
    public function getDevices()
    {
        $assets =  JamfComputer::select('jamf_computers.id', 'jamf_computers.serial_no', 'jamf_computers.model', DB::raw('"Apple" as platform'), 'jamf_computers.os_version', 'jamf_computers.asset_id', 'jamf_computers.checkin_date as last_seen', 'jamf_computers.user_id', 'jamf_computers.user_name', 'jamf_computers.jamf_email as email')
            ->with('asset', 'user')
            ->join('assets', 'assets.serial_no', '=', 'jamf_computers.serial_no');
        return $assets;
    }

    /**
     * Filter with os version in operating system compliance settings
     * In the settings area we sets the platform and current OS version of different Operating Systems.
     * @param Query $asstes
     * @param OperatingSystemComplianceSettings $settings
     */
    public function filterWithOsComplianceSettings($assets, $settings)
    {
        if ($settings) {
            $assets->where(function ($query) use ($settings) {
                $query->where('os_version', '<', optional($settings)->os_version);
            });
        }

        return $assets;
    }

    /**
     * Making query to filter apple mobile phones and tablets
     * @param Query $assets
     */
    public function filterMobileDevices($assets)
    {
        $assetTypeIds = AssetType::whereIn('slug', ['mobile_phone', 'tablet'])->get()->modelKeys();
        if ($assetTypeIds) {
            $assets->where(function ($query) use ($assetTypeIds) {
                $query->whereIn('assets.asset_type_id', $assetTypeIds)
                    ->orWhere('model', 'LIKE', 'iPhone%')
                    ->orWhere('model', 'LIKE', 'iPad%');
            });
        }

        return $assets;
    }

    /**
     * Making query for filtering android devices
     * @param Query $assets
     */
    public function filterAndroidDevices($assets)
    {
        return $assets->whereRaw('LOWER(model) LIKE ?', ['%android%']);
    }

    /**
     * Making the query for filtering from apple devices
     */
    public function filterAppleDevices($assets, $inputData)
    {
        $assets = $assets->RelationData($inputData['mdm_serial_no'], 'jamf_computers.serial_no');
        $assets = $assets->RelationData($inputData['model'], 'jamf_computers.model');
        $assets = $assets->RelationData($inputData['user'], 'jamf_computers.jamf_email');
        $assets = $this->jamfRepository->filterWithCheckIn($assets, $inputData['checkinFrom'], $inputData['checkinTo']);
        // $assets = $assets->RelationData($inputData['platform'], 'jamf_computers.platform');
        $assets = $assets->RelationData($inputData['os_version'], 'jamf_computers.os_version');
        $assets = $this->filterAssets($assets, $inputData);

        return $assets;
    }


    /**
     * Making query for Jamf devices with outdated os version.
     */
    public function getDevicesWithOutdatedOSFromJamf()
    {
        $settings = OperatingSystemComplianceSettings::select('os_version')->where('device_type', 'windows')->first();
        $assets = JamfComputer::select('jamf_computers.id', 'jamf_computers.asset_id', 'jamf_computers.os_version', 'jamf_computers.model')
            ->with('asset', 'user')
            ->join('assets', 'assets.serial_no', '=', 'jamf_computers.serial_no');
        $assets = $this->filterWithOutdatedOS($assets, $settings);
        return $assets;
    }

    /**
     * fetch Jamf filter values for report
     * @param string $type Workspace ONE attribute name
     */
    public function getFilterData($type)
    {
        if ($type == 'platform') {
            return [];
        }
        return JamfComputer::select("$type")->groupBy($type)->get()->filter();
    }
}
