<?php

namespace App\Repositories\OsCompliance;

use App\Models\AirWatch;
use App\Models\AssetType;
use App\Models\OperatingSystemComplianceSettings;
use Illuminate\Support\Str;

class AbstractOsComplianceRepository
{

    /**
     * Filter with os version or platform in operating system compliance settings
     * In the settings area we sets the platform and current OS version of different Operating Systems.
     * @param Query $asstes
     * @param OperatingSystemComplianceSettings $settings
     */
    public function filterWithOsComplianceSettings($assets, $settings)
    {
        if ($settings) {

            $assets->where(function ($query) use ($settings) {
                $query->where('platform', '<>', optional($settings)->os_platform)
                    ->orWhere('os_version', '<>', optional($settings)->os_version ?: 0);
            });
        }

        return $assets;
    }

    /**
     * Making query to filter apple mobile phones and tablets
     * @param Query $assets
     */
    public function filterMobileDevices($assets)
    {
        $assetTypeIds = AssetType::whereIn('slug', ['mobile_phone', 'tablet'])->get()->modelKeys();
        if ($assetTypeIds) {
            $assets->where(function ($query) use ($assetTypeIds) {
                $query->whereIn('assets.asset_type_id', $assetTypeIds)
                    ->orWhere('model', 'LIKE', 'iPhone%')
                    ->orWhere('model', 'LIKE', 'iPad%');
            })
                ->where('platform', '<>', 'Android')
                ->where('platform', '<>', 'mac');
        }

        return $assets;
    }


    /**
     * Making query to filter out mobile devices from apple report
     * @param Query $assets
     */
    public function filterAppleComputerDevices($assets)
    {
        $assetTypeIds = AssetType::whereIn('slug', ['mobile_phone', 'tablet'])->get()->modelKeys();
        if ($assetTypeIds) {
            $assets->where(function ($query) use ($assetTypeIds) {
                $query->whereNotIn('assets.asset_type_id', $assetTypeIds)
                    ->where('model', 'NOT LIKE', 'iPhone%')
                    ->where('model', 'NOT LIKE', 'iPad%');
            });
        }

        return $assets;
    }

    /**
     * compare with os version in operating system compliance settings.
     *
     * @param Query                             $asstes
     * @param OperatingSystemComplianceSettings $settings
     */
    public function filterWithOutdatedOS($assets, $settings)
    {

        $assets = $assets->get()->filter(function ($asset) use ($settings) {
            return version_compare($asset->os_version, optional($settings)->os_version) < 0;
        });

        return $assets;
    }


    /**
     * Making the query for asset filters
     */
    public function filterAssets($assets, $inputData)
    {
        $assets     = $assets->relationData($inputData['asset_type_id'], 'assets.asset_type_id');
        $assets     = $assets->relationData($inputData['user_id'], 'assets.user_id');
        $assets     = $assets->relationData($inputData['serial_no'], 'assets.serial_no');
        $assets     = $assets->relationData($inputData['location_id'], 'assets.location_id');
        $assets     = $assets->relationData($inputData['technical_spec_id'], 'assets.technical_spec_id');
        $assets     = $assets->relationData($inputData['asset_status_id'], 'assets.asset_status_id');
        $assets     = $assets->relationData($inputData['make_and_model_id'], 'assets.make_and_model_id');
        return $assets;
    }


    /**
     * Making query for apple devices with outdated os version.
     *
     * @param OperatingSystemComplianceSettings $settings
     */
    public function getAppleDevicesWithOutdatedOS($settings = '')
    {
        $settings = OperatingSystemComplianceSettings::select('os_version')->where('device_type', 'apple')->first();
        $assets = $this->getDevices();
        $assets = $this->filterAppleComputerDevices($assets);
        $assets = $this->filterWithOutdatedOS($assets, $settings);
        return $assets;
    }

    /**
     * Making query for android devices with outdated os version.
     *
     * @param OperatingSystemComplianceSettings $settings
     */
    public function getAndroidDevicesWithOutdatedOS($settings = '')
    {
        $settings = OperatingSystemComplianceSettings::select('os_version')->where('device_type', 'android')->first();
        $assets = $this->getDevices();
        $assets = $this->filterAndroidDevices($assets);
        $assets = $this->filterWithOutdatedOS($assets, $settings);
        return $assets;
    }

    /**
     * Making query for android devices with outdated os version.
     *
     * @param OperatingSystemComplianceSettings $settings
     */
    public function getAppleMobileDevicesWithOutdatedOS($settings = '')
    {
        $settings = OperatingSystemComplianceSettings::select('os_version')->where('device_type', 'apple_mobile')->first();
        $assets = $this->getDevices();
        $assets = $this->filterMobileDevices($assets);
        $assets = $this->filterWithOutdatedOS($assets, $settings);
        return $assets;
    }
}
