<?php

namespace App\Repositories;

use App\Models\Country;
use App\Models\EmployeePosition;
use App\Models\Department;
use App\Models\Division;
use App\Models\EmployeeType;
use App\User;
use Cache;

class OktaDirectoryRepository
{
    protected $charactersToBeReplaced = ['?' => '•'];

    /**
     * Maps the position ID from the APIs response position value
     * @param mixed $item
     * @param mixed $dbField
     * @param mixed $fileHeader
     *
     * @return [type]
     */
    public function getPositionId($item, $dbField, $fileHeader)
    {
        $position = null;
        if (isset($item->$fileHeader) &&  $item->$fileHeader) {
            $slug = str_slug($item->$fileHeader, '_');
            $position = EmployeePosition::firstOrCreate(['name' => trim($item->$fileHeader)], ['slug'  => trim($slug)]);
        }
        $positionId = $position ? $position->id : null;

        return $positionId;
    }

    /**
     * Maps the department from api to department in teqtivity
     * @param mixed $item
     * @param mixed $dbField
     * @param mixed $fileHeader
     *
     * @return [type]
     */
    public function getDepartmentId($item, $dbField, $fileHeader)
    {
        $department = null;
        if (isset($item->$fileHeader) && $item->$fileHeader) {

            $slug = str_slug($item->$fileHeader, '_');
            $department = Department::firstOrCreate(['slug' => trim($slug)], ['name' => $item->$fileHeader]);
        }
        $departmentId = $department ? $department->id : null;

        return $departmentId;
    }

    /**
     * Maps the employee type from api to employee type in teqtivity
     * @param mixed $item
     * @param mixed $dbField
     * @param mixed $fileHeader
     *
     * @return [type]
     */
    public function getEmployeeTypeId($item, $dbField, $fileHeader)
    {
        $employeeType = null;
        if (isset($item->$fileHeader) && $item->$fileHeader) {
            $slug = str_slug($item->$fileHeader, '_');
            $employeeType = EmployeeType::firstOrCreate(['name' => trim($item->$fileHeader)], ['slug' => trim($slug)]);
        }
        $employeeTypeId = $employeeType ? $employeeType->id : null;

        return $employeeTypeId;
    }

    /**
     * Gets the user status
     * @param mixed $item
     *
     * @return [type]
     */
    public function getStatus($item)
    {
        $userStatusMap = Config('user-status');
        $status = 0;
        if ($item->status) {
            if (in_array(strtoupper(trim($item->status)), $userStatusMap['active'])) {
                $status = 1;
            } elseif (in_array(strtoupper(trim($item->status)), $userStatusMap['suspended'])) {
                $status = 2;
            }
        }

        return $status;
    }

    /**
     * Gets the manager ID of the user
     * @param mixed $item
     * @param mixed $dbField
     * @param mixed $fileHeader
     *
     * @return [type]
     */
    public function getManagerId($item, $dbField, $fileHeader)
    {
        if (isset($item->$fileHeader) && $item->$fileHeader) {
            $user = User::where('email', $item->$fileHeader)->select('id')->first();
            if ($user) {
                return $user->id;
            }
        }

        return null;
    }

    /**
     * Gets the country id of the user
     * @param mixed $item
     * @param mixed $dbField
     * @param mixed $fileHeader
     *
     * @return [type]
     */
    public function getCountryId($item, $dbField, $fileHeader)
    {
        $countryId = null;
        if (isset($item->$fileHeader) && $item->$fileHeader) {
            $countryId = Country::where('country_code', $item->$fileHeader)->pluck('id')->first();
        }
        return $countryId ? $countryId : null;
    }

    /**
     * Gets the last updated date
     * @param mixed $item
     *
     * @return [type]
     */
    public function getUpdatedAtDate($item)
    {
        return general_parse_datetime($item->lastUpdated) ? general_parse_datetime($item->lastUpdated) : null;
    }

    /**
     * Gets the created date
     * @param mixed $item
     *
     * @return [type]
     */
    public function getCreatedAtDate($item)
    {
        return general_parse_datetime($item->created) ? general_parse_datetime($item->created) : null;
    }

    /**
     * Gets last status update date.
     * @param mixed $item
     *
     * @return [type]
     */
    public function getStatusChangedDate($item)
    {
        return general_parse_datetime($item->statusChanged) ? general_parse_datetime($item->statusChanged) : null;
    }

    /**
     * Get the division id
     * @param mixed $item
     * @param mixed $dbField
     * @param mixed $fileHeader
     *
     * @return [type]
     */
    public function getDivisionId($item, $dbField, $fileHeader)
    {
        $divisionId = null;
        if (isset($item->$fileHeader) && $item->$fileHeader) {

            $slug = str_slug($item->$fileHeader, '_');

            $division = Division::firstOrCreate(['name' => trim($item->$fileHeader)], ['slug' => trim($slug)]);
            $divisionId = $division ? $division->id : null;
        }

        return $divisionId;
    }
}
