<?php

namespace App\Repositories;

use App\Models\Location;
use App\Models\ZoomDevice;

class LocationRepository
{
    /**
     * Constructs a new instance of the LocationRepository class.
     *
     * @param ZoomDevice $zoomDeviceModel The Zoom device model instance.
     */
    public function __construct(
        protected ZoomDevice $zoomDeviceModel
    ) {}

    /**
     * get query for locations
     */
    public function getLocations()
    {
        $locations = Location::with(['locationType', 'siteCode', 'region', 'country'])->withCount('assets')
            ->regularLocations();

        return $locations;
    }

    /**
     * get query for filters direct location fields
     * @param $locations
     * @return [type]
     */
    public function filterLocationFields($locations, $field, $value)
    {
        if (!$value) {
            return $locations;
        }
        if (is_array($value)) {
            $locations = $locations->whereIn($field, $value);
        } else {
            $locations = $locations->where($field, $value);
        }
        return $locations;
    }

    /**
     * Filter basd on search text
     * @param mixed $locations
     * @param mixed $searchText
     *
     * @return [type]
     */
    public function searchLocationByKeyword($locations, $searchText)
    {
        if ($searchText) {
            $locations = $locations->where(function ($query) use ($searchText) {
                $query->where('room_name', 'like', '%' . $searchText . '%')
                    ->orWhere('address', 'like', '%' . $searchText . '%')
                    ->orWhere('building', 'like', '%' . $searchText . '%')
                    ->orWhere('floor', 'like', '%' . $searchText . '%');
            })->orWhereHas('locationType', function ($query) use ($searchText) {
                $query->where('location_type_name', 'like', '%' . $searchText . '%');
            })
                ->orWhereHas('siteCode', function ($query) use ($searchText) {
                    $query->where('code', 'like', '%' . $searchText . '%');
                })
                ->orWhereHas('region', function ($query) use ($searchText) {
                    $query->where('name', 'like', '%' . $searchText . '%');
                })
                ->orWhereHas('country', function ($query) use ($searchText) {
                    $query->where('name', 'like', '%' . $searchText . '%');
                });
        }

        return $locations;
    }

    /**
     * Filter location by status
     * @param mixed $locations
     * @param mixed $status
     *
     * @return [type]
     */
    public function filterByLocationStatus($locations, $status)
    {
        if ($status != 'select') {
            if ($status == 1) {
                $locations = $locations->active();
            } elseif ($status == 2) {
                $locations = $locations->inActive();
            }
        }
        return $locations;
    }

    /**
     * Filter location has assets yes / no
     * @param mixed $locations
     * @param mixed $haveAssets
     *
     * @return [type]
     */
    public function filterLocationHasAssets($locations, $haveAssets)
    {
        if ($haveAssets != 'select') {
            if ($haveAssets == 1) {
                $locations = $locations->has('assets');
            } elseif ($haveAssets == 2) {
                $locations = $locations->doesntHave('assets');
            }
        }
        return $locations;
    }

    /**
     * Filter by no of assets in location
     * @param mixed $locations
     * @param mixed $noOfAssets
     *
     * @return [type]
     */
    public function filterLocationByNoofAssets($locations, $noOfAssets)
    {
        if ($noOfAssets != '') {
            $locations = $locations->has('assets', '=', $noOfAssets);
        }
        return $locations;
    }

    /**
     * Filter by location vc or not
     * @param mixed $locations
     * @param mixed $vc
     *
     * @return [type]
     */
    public function filterLocationByVc($locations, $vc)
    {
        if ($vc != 'select') {
            if ($vc == 1) {
                $locations = $locations->vc();
            } elseif ($vc == 2) {
                $locations = $locations->notVc();
            }
        }
        return $locations;
    }

    /**
     * Retrieves Zoom devices associated with a specific location.
     *
     * @param int $locationId The ID of the location.
     * @return \Illuminate\Database\Eloquent\Builder|null The query builder for retrieving Zoom devices or null if no location ID is provided.
     */
    public function getZoomDevices($locationId)
    {
        if (!$locationId) {
            return null;
        }

        return $this->zoomDeviceModel->with('asset:id,serial_no,location_id')
            ->select('id', 'serial_no', 'location_id', 'asset_id')
            ->where('location_id', $locationId);
    }

    /**
     * Filters the given query to include only locations that are (1) or are not (2) zoom locations.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query The query to filter.
     * @param int $zoom The zoom status to filter by (1 = zoom location, 2 = not zoom location).
     *
     * @return \Illuminate\Database\Eloquent\Builder The filtered query.
     */
    public function filterZoom($query, $zoom)
    {
        if ($zoom == 1) {
            $query = $query->zoom();
        } elseif ($zoom == 2) {
            $query = $query->notZoom();
        }

        return $query;
    }
}
