<?php

namespace App\Repositories\InventoryConsumption;

use App\Models\AssetHistory;
use App\Models\AssetType;
use App\Models\AssetStatus;
use App\Models\LocationType;
use Illuminate\Database\Eloquent\Builder;

class ConsumptionReportRepository
{

    protected $assetStatusModel;
    protected $locationTypeModel;
    protected $assetTypeModel;
    protected $assetHistoryModel;

    /**
     * Constructs a new instance of the class.
     *
     * @param AssetStatus $assetStatusModel The asset status model.
     * @param LocationType $locationTypeModel The location type model.
     * @param AssetType $assetTypeModel The asset type model.
     * @param AssetHistory $assetHistoryModel The asset history model.
     */
    public function __construct(AssetStatus $assetStatusModel, LocationType $locationTypeModel, AssetType $assetTypeModel, AssetHistory $assetHistoryModel)
    {
        $this->assetStatusModel = $assetStatusModel;
        $this->locationTypeModel = $locationTypeModel;
        $this->assetTypeModel = $assetTypeModel;
        $this->assetHistoryModel = $assetHistoryModel;
    }



    /**
     * Get consumption data for the report.
     *
     * @return Builder
     */
    public function getConsumptionData()
    {
        $queryConditionsData = $this->getQueryConditionsData();

        // Query to get consumption data
        $consumptionData = $this->assetHistoryModel->with([
                'newUser', 'newUser.employeeType', 'oldStatus', 'newLocation', 'oldLocation', 
                'asset:id,asset_tag,serial_no,created_at,asset_status_id,asset_type_id,make_and_model_id,technical_spec_id', 
                'asset.assetStatus:id,name', 'asset.assetType:id,name', 
                'asset.makeAndModel:id,name,manufacturer_id', 'asset.makeAndModel.manufacturer', 'asset.technicalSpec', 'oldLocation.siteCode.country', 
                'oldLocation.siteCode.region', 'newUser.department', 'newUser.position', 'asset.latestAssetHistory.user'
            ])
            ->select('asset_histories.*')
            ->whereIn('new_asset_status_id', $queryConditionsData['allowedStatusId'])
            ->join('locations as old_locations', function($query) use ($queryConditionsData){
                $query->on('old_locations.id', '=', 'asset_histories.old_location_id')
                ->where('old_locations.location_type_id', $queryConditionsData['storageType']);
            })
            ->whereHas('asset', function ($query) use ($queryConditionsData) {
                $query->whereIn('asset_type_id', $queryConditionsData['assetTypeIds']);
            });

        return $consumptionData;
    }

    /**
     * Get the count of consumptionData for the last 12 months based on created_at of AssetHistory.
     * @param array $queryConditionsData
     * @param string $startDate
     * @param string $endDate
     *
     * @return object
     */
    public function getCountOfConsumptionTrend($queryConditionsData, $startDate, $endDate)
    {
            // Query to get count of consumption data for each month.
            $assets = $this->assetHistoryModel->whereIn('new_asset_status_id', $queryConditionsData['allowedStatusId'])
                ->join('locations as old_locations', function($query) use ($queryConditionsData){
                    $query->on('old_locations.id', '=', 'asset_histories.old_location_id')
                    ->where('old_locations.location_type_id', $queryConditionsData['storageType']);
                })
                ->whereHas('asset', function ($query) use ($queryConditionsData) {
                    $query->whereIn('asset_type_id', $queryConditionsData['assetTypeIds']);
                })
                ->whereBetween('asset_histories.created_at', [$startDate, $endDate]);

            return $assets;
    }

    /**
     * get conditional data for the query
     * @param array $queryConditionsData
     */
    public function getQueryConditionsData()
    {
        // Get allowed status id.
        $queryConditionsData['allowedStatusId'] = $this->assetStatusModel->whereIn('slug', config('reports.consumption_report.include_statuses'))->pluck('id');

        // Get storage location type id.
        $queryConditionsData['storageType'] = $this->locationTypeModel->where('location_type_name', 'Storage')->first()->id;

        // Get asset type ids.
        $queryConditionsData['assetTypeIds'] = $this->assetTypeModel->whereIn('slug', config('reports.consumption_report.include_asset_types'))->pluck('id');

        return $queryConditionsData;
    }
}
