<?php

namespace App\Repositories;

use App\Models\Asset;
use App\Models\Manufacturer;

class InStockComputerRepository
{

    protected $assetModel;
    protected $manufacturerModel;

    /**
     * Constructs a new instance of the class.
     *
     * @param Asset $assetModel The asset model.
     * @param Manufacturer $manufacturerModel The manufacturer model.
     */
    public function __construct(Asset $assetModel, Manufacturer $manufacturerModel)
    {
        $this->assetModel = $assetModel;
        $this->manufacturerModel = $manufacturerModel;
    }



    /**
     * get query for all instock computers
     * @return [type]
     */
    public function getInstockComputers()
    {
        $assets = $this->assetModel->select(
                'id','asset_type_id', 'asset_status_id', 'make_and_model_id', 'location_id', 'technical_spec_id',
                'asset_tag', 'serial_no', 'po_id', 'warranty_end_date','created_at',
            )
            ->with(
                'location:id,room_name', 'assetType:id,name,slug', 'makeAndModel:id,name,manufacturer_id', 'makeAndModel.manufacturer:id,name,slug', 
                'technicalSpec:id,details', 'assetStatus:id,name,slug', 'latestAssetHistory:updated_at',
                'latestAssetHistory.user:first_name,last_name'
            )
            ->whereHas('assetType', function ($query) {
                $query->whereIn('slug', config('reports.in-stock-computers.include_asset_types'));
            })
            ->whereHas('assetStatus', function ($query) {
                $query->whereIn('slug', config('reports.in-stock-computers.include_statuses'));
            });

        return $assets;
    }

    /**
     * get manufacturer id(s) based on computer type
     */
    public function getManufacturerByType($type)
    {
        if ($type == "pc") {
            return $this->manufacturerModel->distinct()
                ->whereIn('slug', config('manufacturer-asset-types.pc'))
                ->whereHas('makeAndModel', function ($query) {
                    $query->where('slug', 'not like', '%chromebook%')
                        ->whereHas('assetType', function ($query) {
                            $query->whereIn('slug', ['desktop', 'laptop', 'computer']);
                        });
                })->pluck('id')->toArray();
        } elseif ($type == "chromebook") {
            return $this->manufacturerModel->whereHas('makeAndModel', function ($query) {
                $query->where('slug', 'like', '%chromebook%')
                    ->whereHas('assetType', function ($query) {
                        $query->whereIn('slug', ['desktop', 'laptop', 'computer']);
                    });
            })->pluck('id')->toArray();
        } else {
            return $this->manufacturerModel->where('slug', 'apple')
                ->whereHas('makeAndModel', function ($query) {
                    $query->where('slug', 'not like', '%chromebook%')
                        ->whereHas('assetType', function ($query) {
                            $query->whereIn('slug', ['desktop', 'laptop', 'computer']);
                        });
                })->pluck('id')->toArray();
        }
    }
}
