<?php

namespace App\Repositories;

use App\User;
use App\Models\Country;
use App\Models\Location;
use App\Models\Department;
use App\Models\EmployeeType;
use App\Models\EmployeePosition;

class HiBobRepository
{

    /**
     * Gets the location ID from the given file header value.
     * Searches for a location with the given room name and returns the location ID if found.
     *
     * @param mixed $fileHeader
     *
     * @return int|null The location ID or null if not found.
     */
    public function getLocationId($fileHeader)
    {
        $location = null;
        if (!empty($fileHeader)) {
            $location = Location::where('room_name', trim($fileHeader))->first();
        }
        $locationId = $location ? $location->id : null;
        return $locationId;
    }

    /**
     * Gets the country id of the user
     * @param string $fileHeader the value of the country field in the file
     * @return int|null The country id or null if not found.
     */
    public function getCountryId($fileHeader)
    {
        $country = null;
        if (!empty($fileHeader)) {
            $fileHeader = $fileHeader == "United States of America" ? "United States" : $fileHeader;
            $country = Country::where('name', trim($fileHeader))->first();
        }
        $countryId = $country ? $country->id : null;
        return $countryId;
    }

    /**
     * Maps the position ID from the APIs response position value
     * @param mixed $fileHeader
     *
     * @return int|null
     */
    public function getPositionId($fileHeader)
    {
        $position = null;
        if (!empty($fileHeader)) {
            $slug = str_slug($fileHeader, '_');
            $position = EmployeePosition::firstOrCreate(['name' => $fileHeader], ['slug' => trim($slug)]);
        }
        $positionId = $position ? $position->id : null;
        return $positionId;
    }

    /**
     * Maps the department ID from the APIs response department value
     * @param mixed $fileHeader
     *
     * @return int|null
     */
    public function getDepartmentId($fileHeader)
    {
        $department = null;
        if (!empty($fileHeader)) {
            $department = Department::firstOrCreate(['name' => trim($fileHeader)]);
        }
        $departmentId = $department ? $department->id : null;
        return $departmentId;
    }

    /**
     * Maps the employee type ID from the APIs response employee type value
     * @param mixed $fileHeader
     *
     * @return int|null
     */
    public function getEmployeeTypeId($fileHeader)
    {
        $employeeType = null;
        if (!empty($fileHeader)) {
            $slug = str_slug($fileHeader, '_');
            $employeeType = EmployeeType::firstOrCreate(['name' => $fileHeader], ['slug' => trim($slug)]);
        }
        $employeeTypeId = $employeeType ? $employeeType->id : null;
        return $employeeTypeId;
    }

    /**
     * Retrieves the manager ID based on the given file header value.
     * Searches for a user with the specified employee ID and returns the user ID if found.
     *
     * @param mixed $fileHeader
     *
     * @return int|null The manager ID or null if not found.
     */
    public function getManagerId($fileHeader)
    {
        if ($fileHeader) {
            $user = User::where('employee_id', $fileHeader)->select('id')->first();
            if ($user) {
                return $user->id;
            }
        }

        return null;
    }

    /**
     * Retrieves the status based on the given file header value.
     * Searches for the trimmed value in the list of active states in the hibob-directory config.
     * If found, returns 1 (active), otherwise returns 0 (inactive).
     *
     * @param mixed $fileHeader
     *
     * @return int The status (1 = active, 0 = inactive).
     */
    public function getStatus($fileHeader)
    {
        $status = 0;
        if (!empty($fileHeader)) {
            if (in_array(trim($fileHeader), config('hibob-directory.active_states'))) {
                $status = 1;
            }
        }
        return $status;
    }
}
