<?php

namespace App\Repositories;

use App\User;
use App\Models\AssetType;
use App\Models\Location;
use Facades\App\Models\MakeAndModel;
use Illuminate\Support\Facades\DB;

class HardwareStandards
{
	public function getHardwareStandardsAccessories()
	{
		return DB::table('make_and_models')
			->join('manufacturers', 'manufacturers.id', '=', 'make_and_models.manufacturer_id')
			->orderBy('manufacturers.slug')
			->where(
				'make_and_models.asset_type_id',
				AssetType::getBySlug('accessories')->first()->id
			)
			->select(
				'make_and_models.id as id',
				'manufacturers.name as manufacturer',
				'make_and_models.name as makeAndModel',
				DB::raw('CONCAT(manufacturers.name," - ",make_and_models.name) as hardware')
			)
			->get();
	}

	public function getAllHardwareStandards()
	{
		return DB::table('technical_specs')
			->join('make_and_models', 'make_and_models.id', '=', 'technical_specs.make_and_model_id')
			->join('manufacturers', 'manufacturers.id', '=', 'make_and_models.manufacturer_id')
			->orderBy('manufacturers.slug')
			->select('technical_specs.id', 'manufacturers.name as manufacturer_name', 'make_and_models.name as make_and_model_name')
			->get();
	}

	public function getHardwareStandards($assettypeid = null)
	{

		if ($assettypeid == null) {
			return DB::table('make_and_models')
				->join('manufacturers', 'manufacturers.id', '=', 'make_and_models.manufacturer_id')
				->orderBy('manufacturers.name')
				->select(
					'make_and_models.id',
					DB::raw('CONCAT(manufacturers.name," - ",make_and_models.name) as hardware')
				)
				->get();
		} else {
			return DB::table('make_and_models')
				->join('manufacturers', 'manufacturers.id', '=', 'make_and_models.manufacturer_id')
				->orderBy('manufacturers.name')
				->where('make_and_models.asset_type_id', $assettypeid)
				->select(
					'make_and_models.id',
					DB::raw('CONCAT(manufacturers.name," - ",make_and_models.name) as hardware')
				)
				->get();
		}
	}

	public function getHardwareStandardsFilter($assettypeid = null, $type = null)
	{
		$query = MakeAndModel::with('manufacturer');
		if ($type == 'it') {
			$query = $query->getItModels();
		}

		if ($type == 'avassets') {
			$query = $query->getAv();
		}

		if ($type == 'mobile') {
			$query = $query->getMobile();
		}

		if ($type == 'network') {
			$query = $query->getNetwork();
		}

		if ($assettypeid == null) {
			return $query;
		}

		return $query->getMakeModelFromAssetTypes($assettypeid);
	}

	public function getHardwareStandardsCreate($assettypeid = null, $type = null)
	{
		$query = MakeAndModel::with('manufacturer')->has('technicalSpecs');
		if ($type == 'it') {
			$query = $query->getItModels();
		}

		if ($type == 'avassets') {
			$query = $query->getAv();
		}

		if ($type == 'mobile') {
			$query = $query->getMobile();
		}

		if ($type == 'network') {
			$query = $query->getNetwork();
		}

		if ($type == 'research') {
			$query = $query->getResearch();
		}

		if ($assettypeid == null) {
			return $query;
		}

		return $query->getMakeModelFromAssetTypes($assettypeid);
	}

	public function getHardwareStandardsBySlug($slug)
	{
		return DB::table('make_and_models')
			->join('manufacturers', 'manufacturers.id', '=', 'make_and_models.manufacturer_id')
			->orderBy('manufacturers.name')
			->where(
				'make_and_models.asset_type_id',
				AssetType::getBySlug($slug)->first()->id
			)
			->select(
				'make_and_models.id',
				DB::raw('CONCAT(manufacturers.name," - ",make_and_models.name) as hardware')
			)
			->get();
	}

	public function getHardwareStandardsByTechSpec()
	{
		return DB::table('make_and_models')
			->join('technical_specs', 'technical_specs.make_and_model_id', '=', 'make_and_models.id')
			->join('manufacturers', 'manufacturers.id', '=', 'make_and_models.manufacturer_id')
			->join('asset_types', 'asset_types.id', '=', 'make_and_models.asset_type_id')
			->orderBy('asset_types.name')

			->select('asset_types.name as assetName', 'manufacturers.name as manufactureName', 'make_and_models.name as makeAndModelName', 'make_and_models.id as makeAndModelId', 'technical_specs.details as details', 'technical_specs.original_value as original_value', 'technical_specs.new_value as new_value', 'technical_specs.life_of_asset as life_of_asset', 'technical_specs.id as id', 'technical_specs.status as status')
			->get();
	}

	public function getAllTechSpecWithHardwares()
	{
		return DB::table('make_and_models')
			->join('technical_specs', 'technical_specs.make_and_model_id', '=', 'make_and_models.id')
			->join('manufacturers', 'manufacturers.id', '=', 'make_and_models.manufacturer_id')
			->join('asset_types', 'asset_types.id', '=', 'make_and_models.asset_type_id')
			->leftjoin('assets', 'assets.technical_spec_id', '=', 'technical_specs.id')
			->groupBy('technical_specs.id')
			->orderBy('make_and_models.slug')

			->select('asset_types.name as assetName', 'manufacturers.name as manufactureName', 'make_and_models.name as makeAndModelName', 'make_and_models.id as makeAndModelId', 'technical_specs.details as details', 'technical_specs.original_value as original_value', 'technical_specs.new_value as new_value', 'technical_specs.life_of_asset as life_of_asset', 'technical_specs.id as id', 'technical_specs.status as status', DB::raw('COUNT(assets.technical_spec_id) as assets_count'));
	}

	public function getStatus()
	{
		return DB::table('make_and_models')
			->join('technical_specs', 'technical_specs.make_and_model_id', '=', 'make_and_models.id')
			->join('manufacturers', 'manufacturers.id', '=', 'make_and_models.manufacturer_id')
			->join('asset_types', 'asset_types.id', '=', 'make_and_models.asset_type_id')
			->orderBy('make_and_models.slug')
			->select('technical_specs.id as id', 'technical_specs.status as status')
			->get();
	}

	public function getHardwareStandardByAsset()
	{
		return DB::table('make_and_models')
			->join('manufacturers', 'manufacturers.id', '=', 'make_and_models.manufacturer_id')
			->join('asset_types', 'asset_types.id', '=', 'make_and_models.asset_type_id')
			->orderBy('manufacturers.slug')
			->select(
				'make_and_models.id as id',
				'manufacturers.name as manufacturer',
				'make_and_models.name as makeAndModel',
				DB::raw('CONCAT(manufacturers.name," - ",make_and_models.name) as hardware'),
				'asset_types.name as assetType',
				'make_and_models.status as status'
			)
			->get();
	}

	public function getHardwareStatus()
	{
		return DB::table('make_and_models')
			->join('manufacturers', 'manufacturers.id', '=', 'make_and_models.manufacturer_id')
			->join('asset_types', 'asset_types.id', '=', 'make_and_models.asset_type_id')
			->orderBy('manufacturers.slug')
			->select('make_and_models.id as id', 'make_and_models.status as status')
			->get();
	}


	public function getHardwareStandardsSearch()
	{
		return DB::table('make_and_models')
			->join('manufacturers', 'manufacturers.id', '=', 'make_and_models.manufacturer_id')
			->join('asset_types', 'asset_types.id', '=', 'make_and_models.asset_type_id')
			->orderBy('manufacturers.slug');
	}

	public function getHardwareStandardsIt()
	{
		return DB::table('make_and_models')

			->join('manufacturers', 'manufacturers.id', '=', 'make_and_models.manufacturer_id')
			->orderBy('manufacturers.slug')
			->whereIn(
				'make_and_models.asset_type_id',
				AssetType::itAsset()->pluck('id')
			)
			->select(
				'make_and_models.id as id',
				'manufacturers.name as manufacturer',
				'make_and_models.name as makeAndModel',
				DB::raw('CONCAT(manufacturers.name," - ",make_and_models.name) as hardware')
			)
			->get();
	}

	public function getHardwareStandardsServer()
	{
		return DB::table('make_and_models')
			->join('manufacturers', 'manufacturers.id', '=', 'make_and_models.manufacturer_id')
			->orderBy('manufacturers.slug')
			->whereIn('asset_type_id', AssetType::whereIn('slug', ['servers', 'server_rack', 'server_components'])->pluck('id'))
			->select(
				'make_and_models.id as id',
				'manufacturers.name as manufacturer',
				'make_and_models.name as makeAndModel',
				DB::raw('CONCAT(manufacturers.name," - ",make_and_models.name) as hardware')
			)
			->get();
	}

	public function getHardwareStandardsMobile()
	{
		return DB::table('make_and_models')
			->join('manufacturers', 'manufacturers.id', '=', 'make_and_models.manufacturer_id')
			->orderBy('manufacturers.slug')

			->whereIn('asset_type_id', AssetType::mobileAsset()->pluck('id'))
			->select(
				'make_and_models.id as id',
				'manufacturers.name as manufacturer',
				'make_and_models.name as makeAndModel',
				DB::raw('CONCAT(manufacturers.name," - ",make_and_models.name) as hardware')
			)
			->get();
	}

	public function getHardwareStandardsNetwork()
	{
		return DB::table('make_and_models')
			->join('manufacturers', 'manufacturers.id', '=', 'make_and_models.manufacturer_id')
			->orderBy('manufacturers.slug')

			->whereIn('asset_type_id', AssetType::networkAsset()->pluck('id'))
			->select(
				'make_and_models.id as id',
				'manufacturers.name as manufacturer',
				'make_and_models.name as makeAndModel',
				DB::raw('CONCAT(manufacturers.name," - ",make_and_models.name) as hardware')
			)
			->get();
	}

	public function getHardwareStandardsList()
	{
		return DB::table('make_and_models')
			->join('manufacturers', 'manufacturers.id', '=', 'make_and_models.manufacturer_id')
			->orderBy('manufacturers.slug')
			->select(
				'make_and_models.id as id',
				'manufacturers.name as manufacturer',
				'make_and_models.name as makeAndModel',
				DB::raw('CONCAT(manufacturers.name," - ",make_and_models.name) as hardware')
			)
			->get();
	}

	public function getHardwareStandardsAv()
	{
		return DB::table('make_and_models')
			->join('manufacturers', 'manufacturers.id', '=', 'make_and_models.manufacturer_id')
			->orderBy('manufacturers.slug')

			->whereIn('asset_type_id', AssetType::avAsset()->pluck('id'))
			->select(
				'make_and_models.id as id',
				'manufacturers.name as manufacturer',
				'make_and_models.name as makeAndModel',
				DB::raw('CONCAT(manufacturers.name," - ",make_and_models.name) as hardware')
			)
			->get();
	}

	public function getHardwareStandardsResearch()
	{
		return DB::table('make_and_models')

			->join('manufacturers', 'manufacturers.id', '=', 'make_and_models.manufacturer_id')
			->orderBy('manufacturers.slug')
			->whereIn(
				'make_and_models.asset_type_id',
				AssetType::researchAsset()->pluck('id')
			)
			->select(
				'make_and_models.id as id',
				'manufacturers.name as manufacturer',
				'make_and_models.name as makeAndModel',
				DB::raw('CONCAT(manufacturers.name," - ",make_and_models.name) as hardware')
			)

			->get();
	}

	public function getHardwareStandardsWithoutAccessories()
	{
		return DB::table('make_and_models')
			->join('manufacturers', 'manufacturers.id', '=', 'make_and_models.manufacturer_id')
			->orderBy('manufacturers.slug')
			->where(
				'make_and_models.asset_type_id',
				'!=',
				AssetType::getBySlug('accessories')->first()->id
			)
			->select(
				'make_and_models.id as id',
				'manufacturers.name as manufacturer',
				'make_and_models.name as makeAndModel',
				DB::raw('CONCAT(manufacturers.name," - ",make_and_models.name) as hardware')
			)
			->get();
	}

	/** get h/w standards by asset types
	 * @param array $assettypeids
	 *
	 * @return [type]
	 */
	public function getHardwareStandardsByAssetTypes($assettypeids = [])
	{
		return DB::table('make_and_models')
			->join('manufacturers', 'manufacturers.id', '=', 'make_and_models.manufacturer_id')
			->orderBy('manufacturers.name')
			->whereIn('make_and_models.asset_type_id', $assettypeids)
			->select(
				'make_and_models.id as id',
				'manufacturers.name as manufacturer',
				'make_and_models.name as makeAndModel',
				DB::raw('CONCAT(manufacturers.name," - ",make_and_models.name) as hardware')
			)
			->get();
	}

	/**
	 * getHardware With Manufacturer AndAssetType
	 * @param null $manufacturerId
	 * @param null $assetTypeId
	 *
	 * @return eloquent
	 */
	public function getHardwareWithManufacturerAndAssetType($manufacturerId = null, $assetTypeId = null)
	{
		$query = MakeAndModel::with('manufacturer');
		if ($manufacturerId) {
			if (is_array($manufacturerId)) {
				$query = $query->whereIn('manufacturer_id', $manufacturerId);
			} else {
				$query = $query->where('manufacturer_id', $manufacturerId);
			}
		}

		if ($assetTypeId) {
			if (is_array($assetTypeId)) {
				$query = $query->whereIn('asset_type_id', $assetTypeId);
			} else {
				$query = $query->where('asset_type_id', $assetTypeId);
			}
		}
		return $query;
	}
}
