<?php

namespace App\Repositories\DiscoveryTools;

use App\Models\AssetStatus;
use App\Models\AirWatch;
use App\Models\Asset;
use App\Models\AssetType;
use App\Models\EmployeePosition;
use App\Models\Manufacturer;
use App\User;
use Carbon\Carbon;
use DB;

class WorkspaceOneRepository extends AbstractRepository
{
    /**
     * Making query for Assets not created report
     *
     * @return DB Query
     */
    public function getAssetsNotCreated()
    {
        $assets =  AirWatch::select('air_watches.*')
            ->with(
                'asset',
                'mobileCarrier:id,name,country_id',
                'mobileCarrier.country:id,name',
                'user:id,first_name,last_name,email,status,department_id',
                'user.department:id,name',
                'asset.assetStatus',
                'asset.makeAndModel',
                'asset.assetType',
                'asset.makeAndModel.manufacturer',
                'asset.location:id,room_name',
                'asset.user:id,first_name,last_name,email,status,department_id,city',
                'asset.user.department:id,name',
                'asset.latestAssetHistory:id,asset_id,updated_at'
            )
            ->leftJoin('assets', 'assets.serial_no', '=', 'air_watches.serial_no')
            ->whereNull('assets.serial_no');
        return $assets;
    }

    /**
     * Making query for Devices not Active report
     *
     * @return DB Query
     */
    public function getInactiveDevices()
    {
        $statusId   = AssetStatus::whereIn('slug', ['assigned', 'installed'])->get()->pluck('id');
        $query      = AirWatch::with(
            'asset',
            'mobileCarrier:id,name,country_id',
            'mobileCarrier.country:id,name',
            'user:id,first_name,last_name,email,status,department_id',
            'user.department:id,name',
            'asset.assetStatus',
            'asset.makeAndModel',
            'asset.assetType',
            'asset.makeAndModel.manufacturer',
            'asset.location:id,room_name',
            'asset.user:id,first_name,last_name,email,status,department_id,city',
            'asset.user.department:id,name',
            'asset.latestAssetHistory:id,asset_id,updated_at'
        )
            ->whereHas('asset', function ($query) use ($statusId) {
                $query->whereIn('asset_status_id', $statusId);
            })
            ->where('last_seen', '<', \Carbon\Carbon::now()->subDays(30));

        return $query;
    }

    /**
     * Making query for User Mismatch report
     */
    public function getUserMismatchDevices()
    {
        $statusId   = AssetStatus::whereIn('slug', ['assigned'])->get()->pluck('id');
        $assets = AirWatch::with(
            'asset',
            'mobileCarrier:id,name,country_id',
            'mobileCarrier.country:id,name',
            'user:id,first_name,last_name,email,status,department_id',
            'user.department:id,name',
            'asset.assetStatus',
            'asset.makeAndModel',
            'asset.assetType',
            'asset.makeAndModel.manufacturer',
            'asset.location:id,room_name',
            'asset.user:id,first_name,last_name,email,status,department_id,city',
            'asset.user.department:id,name',
            'asset.latestAssetHistory:id,asset_id,updated_at'
        )->select('air_watches.*')->with('asset', 'asset.assetStatus', 'asset.makeAndModel')
            ->join("assets", function ($join) use ($statusId) {
                $join->on('assets.serial_no', '=', 'air_watches.serial_no')
                    ->on('assets.user_id', '<>', 'air_watches.user_id')
                    ->whereIn('assets.asset_status_id', $statusId);
            });

        return $assets;
    }


    /**
     * Making query for User Mismatch report
     */
    public function getDevicesForAllReport()
    {
        $statusId   = AssetStatus::whereIn('slug', ['assigned', 'loaned'])->get()->pluck('id');
        $assets = AirWatch::select('air_watches.*')->with(
            'asset',
            'mobileCarrier:id,name,country_id',
            'mobileCarrier.country:id,name',
            'user:id,first_name,last_name,email,status,department_id',
            'user.department:id,name',
            'asset.assetStatus',
            'asset.makeAndModel',
            'asset.assetType',
            'asset.makeAndModel.manufacturer',
            'asset.location:id,room_name',
            'asset.user:id,first_name,last_name,email,status,department_id,city',
            'asset.user.department:id,name',
            'asset.latestAssetHistory:id,asset_id,updated_at'

        )
            ->leftJoin("assets", function ($join) use ($statusId) {
                $join->on('assets.serial_no', '=', 'air_watches.serial_no')
                    ->whereIn('assets.asset_status_id', $statusId);
            });

        return $assets;
    }

    /**
     * Making query for Devices On Hand Checked In report
     */
    public function getDevicesOnHandCheckedIn()
    {
        $statusId   = AssetStatus::onHandStatuses()->get()->pluck('id');
        $query      = AirWatch::select('air_watches.*')->with(
            'asset',
            'mobileCarrier:id,name,country_id',
            'mobileCarrier.country:id,name',
            'user:id,first_name,last_name,email,status,department_id',
            'user.department:id,name',
            'asset.assetStatus',
            'asset.makeAndModel',
            'asset.assetType',
            'asset.makeAndModel.manufacturer',
            'asset.location:id,room_name',
            'asset.user:id,first_name,last_name,email,status,department_id,city',
            'asset.user.department:id,name',
            'asset.latestAssetHistory:id,asset_id,updated_at'
        )
            ->join("assets", function ($join) {
                $join->on('assets.id', '=', 'air_watches.asset_id');
            })
            ->where('air_watches.last_seen', '>', DB::raw('assets.last_status_update'))
            ->whereIn('assets.asset_status_id', $statusId);
        return $query;
    }


    /**
     * Making query for Mobile Number Report
     */
    public function getMobileNumberData()
    {
        $assets =  AirWatch::select('air_watches.*', 'users.city as work_location')->with(
            'asset',
            'mobileCarrier:id,name,country_id',
            'mobileCarrier.country:id,name',
            'user:id,first_name,last_name,email,status,department_id',
            'user.department:id,name',
            'asset.assetStatus',
            'asset.makeAndModel',
            'asset.assetType',
            'asset.makeAndModel.manufacturer',
            'asset.location:id,room_name',
            'asset.user:id,first_name,last_name,email,status,department_id,city',
            'asset.user.department:id,name',
            'asset.latestAssetHistory:id,asset_id,updated_at'
        )
            ->leftJoin('users', 'users.id', '=', 'air_watches.user_id')
            ->where('phone_number', '<>', '')
            ->whereRaw('air_watches.id IN (SELECT MAX(id) from air_watches WHERE phone_number != "" GROUP BY phone_number)');
        return $assets;
    }

    /**
     * Making query for  'Not checking in, not Managed & deployed' report
     *
     * @return Eloquent
     */
    public function getNotCheckedIn()
    {
        $statusId       = AssetStatus::whereIn('slug', ['loan_or_test', 'assigned'])->get()->pluck('id');
        $assetTypeId    = AssetType::whereIn('slug', ['laptop', 'mobile_phone', 'tablet', 'computer'])->get()->pluck('id');

        $query      =    Asset::select('assets.*')
            ->with(
                'user:id,first_name,last_name,email,status,department_id',
                'user.department:id,name',
                'assetStatus',
                'makeAndModel',
                'carrier:id,name,country_id',
                'carrier.country:id,name',
                'assetType',
                'makeAndModel.manufacturer',
                'location:id,room_name',
                'latestAssetHistory:id,asset_id,updated_at'
            )
            ->whereIn('asset_status_id', $statusId)
            ->whereIn('asset_type_id', $assetTypeId)
            ->whereHas('makeAndModel', function ($query) {
                $query->whereHas('manufacturer', function ($query) {
                    $query->where('name', 'Apple');
                }); // Assuming the manufacturer relationship is correctly defined in the MakeAndModel model.
            })
            ->doesntHave('airwatch')
            ->doesntHave('chromebook')
            ->doesntHave('intune')
            ->doesntHave('jamf')
            ->doesntHave('KandjiDevices')
            ->doesntHave('mobileIron');

        return $query;
    }

    /**
     * Filter with User start Date
     */
    public function filterWithUserStartDate($query, $from, $to)
    {
        $query->whereHas('user', function ($query) use ($from, $to) {
            if ($from) {
                $query->whereDate('hire_date', '>=', convert_to_db_date($from));
            }

            if ($to) {
                $query->whereDate('hire_date', '<=', convert_to_db_date($to));
            }
        });

        return $query;
    }
}
