<?php

namespace App\Repositories\DiscoveryTools;

use App\Models\Asset;
use App\Models\JumpCloudDevices;
use App\Models\AssetStatus;
use App\Models\AssetType;
use Illuminate\Support\Facades\DB;

class JumpCloudRepository extends AbstractRepository
{
    /**
     * Making query for All JumpCloud report
     */
    public function getAllDevices()
    {
        $assets = JumpCloudDevices::with(
            'asset',
            'user:id,first_name,last_name,email,status,department_id',
            'user.department:id,name',
            'asset.assetStatus',
            'asset.makeAndModel',
            'asset.assetType',
            'asset.makeAndModel.manufacturer',
            'asset.location:id,room_name',
            'asset.user:id,first_name,last_name,email,status,department_id,city',
            'asset.user.department:id,name',
            'asset.latestAssetHistory:id,asset_id,updated_at'
        );

        return $assets;
    }

    /**
     * Devices not created
     */
    public function getAssetsNotCreated()
    {
        return JumpCloudDevices::with(
            'asset',
            'user:id,first_name,last_name,email,status,department_id',
            'user.department:id,name',
            'asset.assetStatus',
            'asset.makeAndModel',
            'asset.assetType',
            'asset.makeAndModel.manufacturer',
            'asset.location:id,room_name',
            'asset.user:id,first_name,last_name,email,status,department_id,city',
            'asset.user.department:id,name',
            'asset.latestAssetHistory:id,asset_id,updated_at'
        )->where('asset_id', 0);
    }

    /**
     * Making query for User Mismatch report
     */
    public function getUserMismatchDevices()
    {
        $statusId = AssetStatus::whereIn('slug', ['assigned'])->get()->pluck('id');

        $assets = JumpCloudDevices::with(
            'asset',
            'user:id,first_name,last_name,email,status,department_id',
            'user.department:id,name',
            'asset.assetStatus',
            'asset.makeAndModel',
            'asset.assetType',
            'asset.makeAndModel.manufacturer',
            'asset.location:id,room_name',
            'asset.user:id,first_name,last_name,email,status,department_id,city',
            'asset.user.department:id,name',
            'asset.latestAssetHistory:id,asset_id,updated_at'
        )
            ->select('jump_cloud_devices.*')
            ->join("assets", function ($join) use ($statusId) {
                $join->on('assets.id', '=', 'jump_cloud_devices.asset_id')
                    ->on('assets.user_id', '<>', 'jump_cloud_devices.user_id')
                    ->whereIn('assets.asset_status_id', $statusId);
            });

        return $assets;
    }

    /**
     * Making query for  'Not checking in, not Managed & deployed' report
     */
    public function getNotManaged()
    {
        $statusId    = AssetStatus::whereIn('slug', ['assigned', 'loaned'])->get()->pluck('id');
        $assetTypeId = AssetType::whereIn('slug', ['laptop', 'mobile_phone', 'tablet', 'computer'])->get()->pluck('id');

        $query       = Asset::select('assets.*')
            ->with(
                'user:id,first_name,last_name,email,status,department_id',
                'user.department:id,name',
                'assetStatus',
                'makeAndModel',
                'carrier:id,name,country_id',
                'carrier.country:id,name',
                'assetType',
                'makeAndModel.manufacturer',
                'location:id,room_name',
                'latestAssetHistory:id,asset_id,updated_at'
            )
            ->whereIn('asset_status_id', $statusId)
            ->whereIn('asset_type_id', $assetTypeId)
            ->doesntHave('airwatch')
            ->doesntHave('chromebook')
            ->doesntHave('intune')
            ->doesntHave('jamf')
            ->doesntHave('KandjiDevices')
            ->doesntHave('mobileIron')
            ->doesntHave('jumpCloud');

        return $query;
    }

    /**
     * Making query for DevicesNotActive report
     */
    public function getDevicesNotActive()
    {
        $statusId = AssetStatus::whereIn('slug', ['assigned', 'loaned'])->get()->pluck('id');

        $assets = JumpCloudDevices::with(
            'asset',
            'user:id,first_name,last_name,email,status,department_id',
            'user.department:id,name',
            'asset.assetStatus',
            'asset.makeAndModel',
            'asset.assetType',
            'asset.makeAndModel.manufacturer',
            'asset.location:id,room_name',
            'asset.user:id,first_name,last_name,email,status,department_id,city',
            'asset.user.department:id,name',
            'asset.latestAssetHistory:id,asset_id,updated_at'
        )
            ->where('last_seen', '<', \Carbon\Carbon::now()->subDays(30))
            ->whereHas('asset', function ($query) use ($statusId) {
                $query->whereIn('asset_status_id', $statusId);
            });

        return $assets;
    }

    /**
     * Making query for Devices On Hand Checked In report
     */
    public function getDevicesOnHandCheckedIn()
    {
        $statusId   = AssetStatus::onHandStatuses()->get()->pluck('id');
        $query      = JumpCloudDevices::select('jump_cloud_devices.*')
            ->with(
                'asset',
                'user:id,first_name,last_name,email,status,department_id',
                'user.department:id,name',
                'asset.assetStatus',
                'asset.makeAndModel',
                'asset.assetType',
                'asset.makeAndModel.manufacturer',
                'asset.location:id,room_name',
                'asset.user:id,first_name,last_name,email,status,department_id,city',
                'asset.user.department:id,name',
                'asset.latestAssetHistory:id,asset_id,updated_at'
            )
            ->join("assets", function ($join) {
                $join->on('assets.id', '=', 'jump_cloud_devices.asset_id');
            })
            ->where('jump_cloud_devices.last_seen', '>', DB::raw('assets.last_status_update'))
            ->whereIn('assets.asset_status_id', $statusId);

        return $query;
    }

    /**
     * To populate data on report filters
     * @param mixed $search
     * @param mixed $type
     * 
     * @return [type]
     */
    public function filterField($search, $type)
    {
        $res = JumpCloudDevices::distinct()->select("$type")->where("$type", 'like', '%' . $search . '%')->get();
        $result = $res->mapWithKeys(function ($item, $key) use ($type) {
            return [
                $key => [
                    'id' => $item[$type],
                    'text' => $item[$type],
                ]
            ];
        });

        return $result;
    }
}
