<?php

namespace App\Repositories\DiscoveryTools;

use App\Models\Asset;
use App\Models\JamfComputer;
use App\Models\AssetStatus;
use App\Models\AssetType;
use Illuminate\Support\Facades\DB;

class JamfRepository extends AbstractRepository
{
	public function getAssetsNotCreated()
	{
		return JamfComputer::with(
			'asset',
			'user:id,first_name,last_name,email,status,department_id',
			'user.department:id,name',
			'asset.assetStatus',
			'asset.makeAndModel',
			'asset.assetType',
			'asset.makeAndModel.manufacturer',
			'asset.location:id,room_name',
			'asset.user:id,first_name,last_name,email,status,department_id,city',
			'asset.user.department:id,name',
			'asset.latestAssetHistory:id,asset_id,updated_at'
		)->where('asset_id', 0);
	}


	public function filterField($search, $type)
	{
		return JamfComputer::distinct()->select("$type")->where("$type", 'like', '%' . $search . '%');
	}

	/**
	 * Making query for User Mismatch report
	 */
	public function getUserMismatchDevices()
	{
		$statusId = AssetStatus::whereIn('slug', ['assigned'])->get()->pluck('id');

		$assets = JamfComputer::with(
			'asset',
			'user:id,first_name,last_name,email,status,department_id',
			'user.department:id,name',
			'asset.assetStatus',
			'asset.makeAndModel',
			'asset.assetType',
			'asset.makeAndModel.manufacturer',
			'asset.location:id,room_name',
			'asset.user:id,first_name,last_name,email,status,department_id,city',
			'asset.user.department:id,name',
			'asset.latestAssetHistory:id,asset_id,updated_at'
		)
			->select('jamf_computers.*')
			->join("assets", function ($join) use ($statusId) {
				$join->on('assets.id', '=', 'jamf_computers.asset_id')
					->on('assets.user_id', '<>', 'jamf_computers.user_id')
					->whereIn('assets.asset_status_id', $statusId);
			});

		return $assets;
	}

	/**
	 * Making query for DevicesNotActive report
	 */
	public function getDevicesNotActive()
	{
		$statusId = AssetStatus::whereIn('slug', ['assigned', 'loaned'])->get()->pluck('id');

		$assets = JamfComputer::with(
			'asset',
			'user:id,first_name,last_name,email,status,department_id',
			'user.department:id,name',
			'asset.assetStatus',
			'asset.makeAndModel',
			'asset.assetType',
			'asset.makeAndModel.manufacturer',
			'asset.location:id,room_name',
			'asset.user:id,first_name,last_name,email,status,department_id,city',
			'asset.user.department:id,name',
			'asset.latestAssetHistory:id,asset_id,updated_at'
		)
			->where('checkin_date', '<', \Carbon\Carbon::now()->subDays(30))
			->whereHas('asset', function ($query) use ($statusId) {
				$query->whereIn('asset_status_id', $statusId);
			});

		return $assets;
	}

	public function getDevicesOnHandCheckedIn()
	{
		$statusId   = AssetStatus::onHandStatuses()->get()->pluck('id');
		$assets = JamfComputer::select('jamf_computers.*', 'assets.last_status_update as last_modified_date')
			->with(
				'asset',
				'user:id,first_name,last_name,email,status,department_id',
				'user.department:id,name',
				'asset.assetStatus',
				'asset.makeAndModel',
				'asset.assetType',
				'asset.makeAndModel.manufacturer',
				'asset.location:id,room_name',
				'asset.user:id,first_name,last_name,email,status,department_id,city',
				'asset.user.department:id,name',
				'asset.latestAssetHistory:id,asset_id,updated_at'
			)
			->join("assets", function ($join) {
				$join->on('assets.id', '=', 'jamf_computers.asset_id');
			})
			->where('jamf_computers.checkin_date', '>', DB::raw('assets.last_status_update'))
			->whereIn('assets.asset_status_id', $statusId);

		return $assets;
	}

	public function getAssignAssets()
	{
		return JamfComputer::select('jamf_computers.name as computer_name', 'users.email as email_address', DB::raw('CONCAT(users.last_name, ", ", users.first_name) AS real_name'), 'employee_positions.name as position', 'departments.name as department')
			->join('assets', 'jamf_computers.asset_id', '=', 'assets.id')
			->join('users', 'assets.user_id', '=', 'users.id')
			->leftJoin('employee_positions', 'employee_positions.id', '=', 'users.position_id')
			->leftJoin('departments', 'departments.id', '=', 'users.department_id');
	}

	public function filterRelation($query, $field, $value)
	{
		if ($value == "") {
			return $query;
		}
		if (is_array($value)) {
			return $query->whereIn('jamf_computers.' . $field, $value);
		}

		return $query->where('jamf_computers.' . $field, $value);
	}

	/**
	 * Making query for All jamf report
	 */
	public function getAllJamfDevices()
	{
		$assets = JamfComputer::select('jamf_computers.*', 'asset_histories.updated_at as last_modified_date')
			->with(
				'asset',
				'user:id,first_name,last_name,email,status,department_id',
				'user.department:id,name',
				'asset.assetStatus',
				'asset.makeAndModel',
				'asset.assetType',
				'asset.makeAndModel.manufacturer',
				'asset.location:id,room_name',
				'asset.user:id,first_name,last_name,email,status,department_id,city',
				'asset.user.department:id,name',
				'asset.latestAssetHistory:id,asset_id,updated_at'
			)
			->leftJoin("assets", 'assets.id', '=', 'jamf_computers.asset_id')
			->leftJoin("asset_histories", function ($join) {
				$join->on('asset_histories.asset_id', '=', 'jamf_computers.asset_id')
					->on('asset_histories.id', '=', DB::raw("(SELECT MAX(id) from asset_histories where asset_histories.asset_id = jamf_computers.asset_id)"));
			});

		return $assets;
	}

	/**
	 * Making query for  'Not checking in, not Managed & deployed' report
	 */
	public function getNotManagedDevices()
	{

		$statusId    = AssetStatus::whereIn('slug', ['loan_or_test', 'assigned'])->get()->pluck('id');
		$assetTypeId = AssetType::whereIn('slug', ['laptop', 'mobile_phone', 'tablet', 'desktop', 'computer'])->get()->pluck('id');

		$query       = Asset::select('assets.*')
			->with(
				'user:id,first_name,last_name,email,status,department_id',
				'user.department:id,name',
				'assetStatus',
				'makeAndModel',
				'carrier:id,name,country_id',
				'carrier.country:id,name',
				'assetType',
				'makeAndModel.manufacturer',
				'location:id,room_name',
				'latestAssetHistory:id,asset_id,updated_at'
			)
			->whereIn('asset_status_id', $statusId)
			->whereIn('asset_type_id', $assetTypeId)
			->doesntHave('airwatch')
			->doesntHave('chromebook')
			->doesntHave('intune')
			->doesntHave('jamf')
			->doesntHave('KandjiDevices')
			->doesntHave('mobileIron')
			->whereHas('makeAndModel', function ($query) {
				$query->whereHas('manufacturer', function ($query) {
					$query->where('name', 'Apple');
				}); // Assuming the manufacturer relationship is correctly defined in the MakeAndModel model.
			});

		return $query;
	}


	/**
	 * Filter with last seen from
	 */
	public function filterWithCheckIn($query, $from, $to)
	{
		if ($from) {
			$query = $query->whereDate('checkin_date', '>=', convert_to_db_date($from));
		}

		if ($to) {
			$query = $query->whereDate('checkin_date', '<=', convert_to_db_date($to));
		}

		return $query;
	}

	/**
	 * Retrieves the pre-stage devices from the database.
	 *
	 * @return \Illuminate\Database\Eloquent\Builder
	 */
	public function getPreStageDevices()
	{
		return JamfComputer::with(
			'asset',
			'user:id,first_name,last_name,email,status,department_id',
			'user.department:id,name',
			'asset.assetStatus',
			'asset.makeAndModel',
			'asset.assetType',
			'asset.makeAndModel.manufacturer',
			'asset.location:id,room_name',
			'asset.user:id,first_name,last_name,email,status,department_id,city',
			'asset.user.department:id,name',
			'asset.latestAssetHistory:id,asset_id,updated_at'
		)->where('api_type', 'Prestage');
	}
}
