<?php

namespace App\Repositories\DiscoveryTools;

use App\Models\IntuneComputer;
use App\Models\AssetStatus;
use App\Models\Asset;
use App\Models\AssetType;
use App\Models\Manufacturer;
use App\Repositories\DiscoveryTools\AbstractRepository;
use DB;
use Illuminate\Support\Facades\DB as FacadesDB;

class IntuneRepository extends AbstractRepository
{
	public function getAssetsNotCreated()
	{
		return IntuneComputer::with(
			'asset',
			'user:id,first_name,last_name,email,status,department_id',
			'user.department:id,name',
			'asset.assetStatus',
			'asset.makeAndModel',
			'asset.assetType',
			'asset.makeAndModel.manufacturer',
			'asset.location:id,room_name',
			'asset.user:id,first_name,last_name,email,status,department_id,city',
			'asset.user.department:id,name',
			'asset.latestAssetHistory:id,asset_id,updated_at'
		)->where('asset_id', 0);
	}


	public function filterField($search, $type)
	{
		if ($type == "model") {
			return IntuneComputer::distinct()->select('manufacturer', 'model')->whereRaw('CONCAT(manufacturer, " ", model) LIKE ?', [$search . '%']);
		}

		return IntuneComputer::distinct()->select("$type")->where("$type", 'like', '%' . $search . '%');
	}

	/**
	 * Making query for User Mismatch report
	 */
	public function getUserMismatchDevices()
	{
		$statusId = AssetStatus::whereIn('slug', ['assigned'])->get()->pluck('id');
		$assets = IntuneComputer::with(
			'asset',
			'user:id,first_name,last_name,email,status,department_id',
			'user.department:id,name',
			'asset.assetStatus',
			'asset.makeAndModel',
			'asset.assetType',
			'asset.makeAndModel.manufacturer',
			'asset.location:id,room_name',
			'asset.user:id,first_name,last_name,email,status,department_id,city',
			'asset.user.department:id,name',
			'asset.latestAssetHistory:id,asset_id,updated_at'
		)
			->select('intune_computers.*')
			->join("assets", function ($join) use ($statusId) {
				$join->on('assets.id', '=', 'intune_computers.asset_id')
					->on('assets.user_id', '<>', 'intune_computers.user_id')
					->whereIn('assets.asset_status_id', $statusId);
			});
		return $assets;
	}

	/**
	 * Making query for DevicesNotActive report
	 */
	public function getDevicesNotActive()
	{
		$statusId = AssetStatus::whereIn('slug', ['assigned', 'loaned'])->get()->pluck('id');
		$assets = IntuneComputer::with(
			'asset',
			'user:id,first_name,last_name,email,status,department_id',
			'user.department:id,name',
			'asset.assetStatus',
			'asset.makeAndModel',
			'asset.assetType',
			'asset.makeAndModel.manufacturer',
			'asset.location:id,room_name',
			'asset.user:id,first_name,last_name,email,status,department_id,city',
			'asset.user.department:id,name',
			'asset.latestAssetHistory:id,asset_id,updated_at'
		)
			->where('checkin_date', '<', \Carbon\Carbon::now()->subDays(30))
			->whereHas('asset', function ($query) use ($statusId) {
				$query->whereIn('asset_status_id', $statusId);
			});
		return $assets;
	}

	public function getDevicesOnHandCheckedIn()
	{
		$statusId   = AssetStatus::onHandStatuses()->get()->pluck('id');
		$assets = IntuneComputer::select('intune_computers.*', 'assets.last_status_update as last_modified_date')
			->with(
				'asset',
				'user:id,first_name,last_name,email,status,department_id',
				'user.department:id,name',
				'asset.assetStatus',
				'asset.makeAndModel',
				'asset.assetType',
				'asset.makeAndModel.manufacturer',
				'asset.location:id,room_name',
				'asset.user:id,first_name,last_name,email,status,department_id,city',
				'asset.user.department:id,name',
				'asset.latestAssetHistory:id,asset_id,updated_at'
			)
			->join("assets", function ($join) {
				$join->on('assets.id', '=', 'intune_computers.asset_id');
			})
			->where('intune_computers.checkin_date', '>', DB::raw('assets.last_status_update'))
			->whereIn('assets.asset_status_id', $statusId);

		return $assets;
	}

	public function getAssignAssets()
	{
		return IntuneComputer::select('intune_computers.name as computer_name', 'users.email as email_address', DB::raw('CONCAT(users.last_name, ", ", users.first_name) AS real_name'), 'employee_positions.name as position', 'departments.name as department')
			->join('assets', 'intune_computers.asset_id', '=', 'assets.id')
			->join('users', 'assets.user_id', '=', 'users.id')
			->leftJoin('employee_positions', 'employee_positions.id', '=', 'users.position_id')
			->leftJoin('departments', 'departments.id', '=', 'users.department_id');
	}

	/**
	 * Making query for All intune report
	 */
	public function getAllIntuneDevices()
	{
		$assets = IntuneComputer::select('intune_computers.*')
			->with(
				'asset',
				'user:id,first_name,last_name,email,status,department_id',
				'user.department:id,name',
				'asset.assetStatus',
				'asset.makeAndModel',
				'asset.assetType',
				'asset.makeAndModel.manufacturer',
				'asset.location:id,room_name',
				'asset.user:id,first_name,last_name,email,status,department_id,city',
				'asset.user.department:id,name',
				'asset.latestAssetHistory:id,asset_id,updated_at'
			)
			->leftJoin("assets", 'assets.id', '=', 'intune_computers.asset_id');

		return $assets;
	}

	public function filterRelation($query, $field, $value)
	{
		if ($value == "") {
			return $query;
		}
		if (is_array($value)) {
			return $query->whereIn('intune_computers.' . $field, $value);
		}
		return $query->where('intune_computers.' . $field, $value);
	}


	/**
	 * Making query for  'Not checking in, not Managed & deployed' report
	 */
	public function getNotManaged()
	{
		$statusId       = AssetStatus::whereIn('slug', ['loan_or_test', 'assigned'])->get()->pluck('id');
		$assetTypeId    = AssetType::whereIn('slug', ['laptop', 'mobile_phone', 'tablet', 'desktop', 'computer'])->get()->pluck('id');
		$manufacturerId = Manufacturer::whereIn('slug', ['apple', 'samsung', 'google'])->pluck('id');

		$query       = Asset::select('assets.*')
			->with(
				'user:id,first_name,last_name,email,status,department_id',
				'user.department:id,name',
				'assetStatus',
				'makeAndModel',
				'carrier:id,name,country_id',
				'carrier.country:id,name',
				'assetType',
				'makeAndModel.manufacturer',
				'location:id,room_name',
				'latestAssetHistory:id,asset_id,updated_at'
			)
			->whereIn('asset_status_id', $statusId)
			->whereIn('asset_type_id', $assetTypeId)
			->whereHas('makeAndModel', function ($query) use ($manufacturerId) {
				$query->where('name', 'not like', '%Chromebook%')
					->where('name', 'not like', '%Pixelbook%')
					->where('name', '<>', 'OnePlus 6T')
					->whereNotIn('manufacturer_id', $manufacturerId);
			})
			->doesntHave('airwatch')
			->doesntHave('chromebook')
			->doesntHave('intune')
			->doesntHave('jamf')
			->doesntHave('KandjiDevices')
			->doesntHave('mobileIron')
			->whereHas('makeAndModel', function ($query) {
				$query->whereHas('manufacturer', function ($query) {
					$query->whereIn('name', ['Dell', 'Lenovo']);
				}); // Assuming the manufacturer relationship is correctly defined in the MakeAndModel model.
			});

		return $query;
	}

}
