<?php

namespace App\Repositories\DiscoveryTools;

use App\Models\AssetStatus;
use App\Models\AirWatch;
use App\Models\Chromebook;
use App\Models\Asset;
use App\Models\AssetType;
use App\User;
use Illuminate\Support\Facades\DB;

class ChromebookRepository extends AbstractRepository
{
    /**
     * Making query for Assets not created report
     *
     * @return DB Query
     */
    public function getAssetsNotCreated()
    {
        $assets =  Chromebook::select('chromebooks.*')
            ->with(
                'asset',
                'user:id,first_name,last_name,email,status,department_id',
                'user.department:id,name',
                'asset.assetStatus',
                'asset.makeAndModel',
                'asset.assetType',
                'asset.makeAndModel.manufacturer',
                'asset.location:id,room_name',
                'asset.user:id,first_name,last_name,email,status,department_id,city',
                'asset.user.department:id,name',
                'asset.latestAssetHistory:id,asset_id,updated_at'
            )
            ->leftJoin('assets', 'assets.serial_no', '=', 'chromebooks.serial_no')
            ->whereNull('assets.serial_no');

        return $assets;
    }

    /**
     * Making query for Devices Inactive report
     *
     * @return DB Query
     */
    public function getInactiveDevices()
    {
        $statusId   = AssetStatus::whereIn('slug', ['assigned', 'installed'])->get()->pluck('id');
        $query      = Chromebook::with(
            'asset',
            'user:id,first_name,last_name,email,status,department_id',
            'user.department:id,name',
            'asset.assetStatus',
            'asset.makeAndModel',
            'asset.assetType',
            'asset.makeAndModel.manufacturer',
            'asset.location:id,room_name',
            'asset.user:id,first_name,last_name,email,status,department_id,city',
            'asset.user.department:id,name',
            'asset.latestAssetHistory:id,asset_id,updated_at'
        )
            ->whereHas('asset', function ($query) use ($statusId) {
                $query->whereIn('asset_status_id', $statusId);
            })
            ->where('last_sync', '<', \Carbon\Carbon::now()->subDays(30));

        return $query;
    }

    /**
     * Making query for All report
     */
    public function getDevicesForAllReport()
    {
        $statusId   = AssetStatus::whereIn('slug', ['assigned', 'loan_or_test', 'gifted'])->get()->pluck('id');
        $assets = Chromebook::select('chromebooks.id', 'chromebooks.user_id', 'chromebooks.email', 'chromebooks.serial_no', 'chromebooks.model', 'chromebooks.asset_id', 'chromebooks.last_sync', 'chromebooks.created_at', 'chromebooks.updated_at', 'chromebooks.mac_address', 'chromebooks.host_name')->with('asset', 'asset.assetStatus', 'asset.makeAndModel')
            ->with(
                'asset',
                'user:id,first_name,last_name,email,status,department_id',
                'user.department:id,name',
                'asset.assetStatus',
                'asset.makeAndModel',
                'asset.assetType',
                'asset.makeAndModel.manufacturer',
                'asset.location:id,room_name',
                'asset.user:id,first_name,last_name,email,status,department_id,city',
                'asset.user.department:id,name',
                'asset.latestAssetHistory:id,asset_id,updated_at'
            )
            ->leftJoin("assets", function ($join) use ($statusId) {
                $join->on('assets.serial_no', '=', 'chromebooks.serial_no')
                    ->whereIn('assets.asset_status_id', $statusId);
            });

        return $assets;
    }

    /**
     * Making query for User Mismatch report
     */
    public function getUserMismatchDevices()
    {
        $statusId   = AssetStatus::whereIn('slug', ['assigned'])->get()->pluck('id');
        $assets = Chromebook::select('chromebooks.*')->with(
            'asset',
            'user:id,first_name,last_name,email,status,department_id',
            'user.department:id,name',
            'asset.assetStatus',
            'asset.makeAndModel',
            'asset.assetType',
            'asset.makeAndModel.manufacturer',
            'asset.location:id,room_name',
            'asset.user:id,first_name,last_name,email,status,department_id,city',
            'asset.user.department:id,name',
            'asset.latestAssetHistory:id,asset_id,updated_at'
        )
            ->join("assets", function ($join) use ($statusId) {
                $join->on('assets.serial_no', '=', 'chromebooks.serial_no')
                    ->on('assets.user_id', '<>', 'chromebooks.user_id')
                    ->whereIn('assets.asset_status_id', $statusId);
            });

        return $assets;
    }

    /**
     * Making query for Devices On Hand Checked In report
     */
    public function getDevicesOnHandCheckedIn()
    {
        $statusId   = AssetStatus::onHandStatuses()->get()->pluck('id');
        $query      = Chromebook::select('chromebooks.*')->with(
            'asset',
            'user:id,first_name,last_name,email,status,department_id',
            'user.department:id,name',
            'asset.assetStatus',
            'asset.makeAndModel',
            'asset.assetType',
            'asset.makeAndModel.manufacturer',
            'asset.location:id,room_name',
            'asset.user:id,first_name,last_name,email,status,department_id,city',
            'asset.user.department:id,name',
            'asset.latestAssetHistory:id,asset_id,updated_at'
        )
            ->join("assets", function ($join) {
                $join->on('assets.id', '=', 'chromebooks.asset_id');
            })
            ->where('chromebooks.last_sync', '>', DB::raw('assets.last_status_update'))
            ->whereIn('assets.asset_status_id', $statusId);

        return $query;
    }

    /**
     * Making query for  'Not checking in, not Managed & deployed' report
     */
    public function getNotCheckedIn()
    {
        $statusId    = AssetStatus::whereIn('slug', ['loaned', 'assigned'])->get()->pluck('id');
        $assetTypeId = AssetType::whereIn('slug', ['laptop', 'mobile_phone', 'tablet', 'computer'])->get()->pluck('id');

        $query       = Asset::select('assets.*')
            ->with(
                'user:id,first_name,last_name,email,status,department_id',
                'user.department:id,name',
                'assetStatus',
                'makeAndModel',
                'carrier:id,name,country_id',
                'carrier.country:id,name',
                'assetType',
                'makeAndModel.manufacturer',
                'location:id,room_name',
                'latestAssetHistory:id,asset_id,updated_at'
            )
            ->whereIn('asset_status_id', $statusId)
            ->whereIn('asset_type_id', $assetTypeId)
            ->whereHas('makeAndModel', function ($query) use ($statusId) {
                $query->where('name', 'like', '%Chromebook%')->orWhere('name', 'like', '%Google%');
            })
            ->doesntHave('airwatch')
            ->doesntHave('chromebook')
            ->doesntHave('intune')
            ->doesntHave('jamf')
            ->doesntHave('KandjiDevices')
            ->doesntHave('mobileIron');

        return $query;
    }


    /**
     * Filter with last seen from
     */
    public function filterWithLastSync($query, $from, $to)
    {
        if ($from) {
            $query = $query->whereDate('last_sync', '>=', convert_to_db_date($from));
        }

        if ($to) {
            $query = $query->whereDate('last_sync', '<=', convert_to_db_date($to));
        }

        return $query;
    }
}
