<?php

namespace App\Repositories\DiscoveryTools;

use Illuminate\Support\Facades\DB;

abstract class AbstractRepository
{
    abstract public function getUserMismatchDevices();
    abstract public function getAssetsNotCreated();
    public function filterRelation($query, $field, $value)
    {
        if ($value == "") {
            return $query;
        }
        if (is_array($value)) {
            return $query->whereIn($field, $value);
        }

        return $query->where($field, $value);
    }

    /**
     * Fiter with User Fields
     *
     * @param $query DB Query
     * @param $field Filter filed name
     * @param $value Filter field value
     *
     * @return DB Query
     */
    public function filterWithUserFields($query, $field, $value)
    {
        if ($value == "") {
            return $query;
        }
        return $query->whereHas('user', function ($query) use ($field, $value) {
            if (is_array($value)) {
                $query->whereIn($field, $value);
            } else {
                $query->where($field, $value);
            }
        });
    }

    public function filterWithTeqtivityUserFields($query, $field, $value)
    {
        if ($value == "") {
            return $query;
        }
        return $query->whereHas('Asset.user', function ($query) use ($field, $value) {
            if (is_array($value)) {
                $query->whereIn($field, $value);
            } else {
                $query->where($field, $value);
            }
        });
    }


    /**
     *  Filter with asset fields
     *
     * @param $query DB Query
     * @param $field Filter filed name
     * @param $value Filter field value
     *
     * @return DB Query
     */

    public function filterWithAssetFields($query, $field, $value)
    {
        if ($value == "") {
            return $query;
        }
        return $query->whereHas('asset', function ($query) use ($field, $value) {
            if (is_array($value)) {
                $query->whereIn($field, $value);
            } else {
                $query->where($field, $value);
            }
        });
    }


    public function filterByAssetFields($query, $field, $value)
    {
        if ($value == "") {
            return $query;
        }
        if (is_array($value)) {
            return $query->whereIn('assets.' . $field, $value);
        }

        return $query->where('assets.' . $field, $value);
    }

    public function filterWithAssetMakeAndModelFields($query, $field, $value)
    {
        if ($value == "") {
            return $query;
        }
        return $query->whereHas('asset.makeAndModel', function ($query) use ($field, $value) {
            if (is_array($value)) {
                $query->whereIn($field, $value);
            } else {
                $query->where($field, $value);
            }
        });
    }


    public function filterByMakeAndModelFields($query, $field, $value)
    {
        if ($value == "") {
            return $query;
        }
        return $query->whereHas('makeAndModel', function ($query) use ($field, $value) {
            if (is_array($value)) {
                $query->whereIn($field, $value);
            } else {
                $query->where($field, $value);
            }
        });
    }

    /**
     * Filter with last seen from
     */
    public function filterWithLastCheckIn($query, $from, $to)
    {
        if ($from) {
            $query = $query->whereDate('last_checkin', '>=', convert_to_db_date($from));
        }

        if ($to) {
            $query = $query->whereDate('last_checkin', '<=', convert_to_db_date($to));
        }

        return $query;
    }

    public function filterWithLocationFields($query, $field, $value)
    {
        if ($value == "") {
            return $query;
        }
        return $query->whereHas('asset.location', function ($query) use ($field, $value) {
            if (is_array($value)) {
                $query->whereIn($field, $value);
            } else {
                $query->where($field, $value);
            }
        });
    }

    public function filterCheckInDateFrom($query, $date)
    {
        if (!$date) {
            return $query;
        }
        return $query->whereDate('checkin_date', '>=', convert_to_db_date($date));
    }

    public function filterCheckInDateTo($query, $date)
    {
        if (!$date) {
            return $query;
        }
        return $query->whereDate('checkin_date', '<', convert_to_db_date($date));
    }

    /**
     * Filter with last seen from
     */
    public function filterWithLastSeen($query, $from, $to)
    {
        if ($from) {
            $query = $query->whereDate('last_seen', '>=', convert_to_db_date($from));
        }

        if ($to) {
            $query = $query->whereDate('last_seen', '<=', convert_to_db_date($to));
        }

        return $query;
    }

    /**
     * Filter with Last Modified Date
     */
    public function filterWithLastModifiedDate($query, $from, $to, $relation = 'latestAssetHistory')
    {
        if ($from || $to) {
            $query->whereHas($relation, function ($query) use ($from, $to) {
                if ($from) {
                    $query->whereDate('updated_at', '>=', convert_to_db_date($from));
                }

                if ($to) {
                    $query->whereDate('updated_at', '<=', convert_to_db_date($to));
                }
            });
        }

        return $query;
    }

    /**
     * Filter with user status
     */
    public function filterByUserStatus($query, $status)
    {
        if ($status != '') {
            $query = $query->where('users.status', $status);
        }

        return $query;
    }

    /**
     * Filter with Last seen days count
     */
    public function filterLastSeenDaysCount($query, $count)
    {
        if ($count) {
            $query  = $query->where('last_seen', '<', \Carbon\Carbon::now()->subDays($count));
        }

        return $query;
    }

    /**
     * Filter with manufacturer
     */
    public function filterWithAssetUserStatus($query, $value)
    {
        if ($value != '') {
            $query  = $query->whereHas('asset.user', function ($query) use ($value) {
                if (is_array($value)) {
                    $query->whereIn('status', $value);
                } else {
                    $query->where('status', $value);
                }
            });
        }

        return $query;
    }

    /**
     * Filter the asset is exist in Teqtivity or not
     */
    public function filterTeqtivityAssetOrNot($query, $value)
    {
        if ($value) {
            if ($value == 'yes') {
                $query = $query->has('asset');
            } else if ($value == 'no') {
                $query = $query->doesntHave('asset');
            }
        }

        return $query;
    }

    /**
     * Filter with mobile carrier country
     */
    public function filterWithCarrierCountry($query, $value, $model = 'mobileCarrier')
    {
        if ($value != '') {
            $query = $query->whereHas($model, function ($query) use ($value) {
                if (is_array($value)) {
                    $query->whereIn('country_id', $value);
                } else {
                    $query->where('country_id', $value);
                }
            });
        }

        return $query;
    }


    /**
     * Filter with lock date from
     */
    public function filterWithLockDate($query, $from, $to)
    {
        if ($from) {
            $query = $query->whereDate('locked_date', '>=', convert_to_db_date($from));
        }

        if ($to) {
            $query = $query->whereDate('locked_date', '<=', convert_to_db_date($to));
        }

        return $query;
    }

    /**
     * Filter with locked date
     */
    public function filterWithLockedStatus($query, $value)
    {
        if ($value == "yes") {
            $query  = $query->whereNotNull('locked_date');
        }

        if ($value == "no") {
            $query  = $query->whereNull('locked_date');
        }

        return $query;
    }
}
