<?php

namespace App\Repositories\AssetsHealth;

use App\Models\Asset;
use App\Models\AssetHealthCategory;
use App\Models\AssetHealthHistory;
use App\Models\AssetHealthReportDetail;
use App\Models\AssetHealthReports;
use App\Models\AssetHealthTests;
use DB;

class AssetsHealthDashboardRepository
{
    /**
     * Retrieve all asset health categories.
     *
     * @return \Illuminate\Database\Eloquent\Collection|AssetHealthCategory[]
     */
    public function getAllAssetHealthCategories()
    {
        return AssetHealthCategory::all();
    }

    /**
     * Retrieve an asset health category by its slug.
     *
     * @param  string  $categorySlug  The slug of the category.
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function getAssetHealthCategoryBySlug($categorySlug)
    {
        return AssetHealthCategory::where('slug', $categorySlug);
    }

    /**
     * Retrieve all asset health tests.
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function getAllAssetHealthTests()
    {
        return AssetHealthCategory::with('assetHealthTests');
    }

    /**
     * Get the count of asset health reports for each category.
     *
     * @return array
     */
    public function getCategoryWiseAssetHealthCount()
    {
        $lastHistory = AssetHealthHistory::orderBy('created_at', 'desc')->first();
        $healthCategories = $this->getAllAssetHealthCategories();
        $testCount = [];

        foreach ($healthCategories as $categories) {
            $categoryId = $categories->id;
            $assets = AssetHealthReports::where('category_id', $categoryId);

            if ($lastHistory) {
                $assets->where('history_id', $lastHistory->id);
            }

            $testCount[$categories->category_name] = $assets->sum('test_count');
        }

        return $testCount;
    }

    /**
     * Get the top asset health report issues.
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function getTopAssetHealthReportIssues()
    {
        $latestHistory = AssetHealthHistory::orderBy('created_at', 'desc')->first();

        $ignoredTestIds = AssetHealthTests::whereStatus(0)->pluck('id')->toArray();

        $query = AssetHealthReports::with('assetHealthTest', 'assetHistory', 'category')
            ->orderBy('test_count', 'desc')
            ->limit(5);

        if (count($ignoredTestIds)) {
            $query = $query->whereNotIn('asset_health_reports.test_id', $ignoredTestIds);
        }

        if ($latestHistory) {
            $query = $query->where('history_id', $latestHistory->id);
        }

        return $query;
    }

    /**
     * Get asset health reports by tests for a specific category.
     *
     * @param  int  $categoryId
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function getAssetHealthReportsByTests($categoryId)
    {
        $latestHistory = AssetHealthHistory::orderBy('created_at', 'desc')->first();

        $query = AssetHealthReports::select('asset_health_reports.id', 'test_name', 'test_count', 'slug', 'status', 'asset_health_tests.id as test_id')
            ->join('asset_health_tests', 'asset_health_tests.id', '=', 'asset_health_reports.test_id')
            ->where('asset_health_reports.category_id', $categoryId)
            ->orderBy('test_count', 'desc');

        if ($latestHistory) {
            $query->where('asset_health_reports.history_id', $latestHistory->id);
        }

        return $query;
    }

    /**
     * Get asset health test details by report ID.
     *
     * @param  int  $reportId
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function getAssetHealthTestDetailById($reportId)
    {
        return AssetHealthReports::select('test_id', 'asset_health_reports.id', 'test_count')
            ->with(['assetHealthTest'])
            ->where('asset_health_reports.id', $reportId);
    }

    /**
     * Get asset health ignored test details by report ID.
     *
     * @param  int  $reportId
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function getAssetHealthIgnoredTestDetailById($reportId)
    {
        return AssetHealthReports::select('test_id', 'asset_health_reports.id', 'test_count')
            ->with(['assetHealthTest'])->withCount('ignoredDetails')
            ->where('asset_health_reports.id', $reportId);
    }

    /**
     * Get asset health report details by report ID.
     *
     * @param  int  $reportId
     * @param  bool|null  $ignored  Whether to include ignored details or not (default is false)
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function getAssetHealthReportDetails($reportId, $ignored = false)
    {
        $query = AssetHealthReportDetail::with([
            'asset',
            'asset.assetType',
            'asset.makeAndModel',
            'asset.makeAndModel.manufacturer',
            'asset.technicalSpec',
            'asset.user',
            'asset.location',
            'asset.assetStatus',
            'asset.latestAssetHistory',
            'asset.latestAssetHistory.user',
            'asset.intune',
            'asset.jamf',
            'asset.airwatch',
            'asset.chromebook',
            'asset.kandjiDevices',
            'asset.mobileIron',
        ])->where('asset_health_report_id', $reportId);

        if ($ignored !== null) {
            $query->where('is_ignored', (int) $ignored);
        }

        return $query;
    }


    /**
     * Get the count of assets with reported issues and the total count of assets.
     *
     * This method retrieves the count of assets with reported issues
     * and the total count of assets. It considers only the most recent health
     * history for asset health analysis.
     *
     * @return array ['totalAssetsCount' => int, 'issueAssetsCount' => int]
     */
    public function getIssuesAssetsCount()
    {
        $totalAssetsCount = Asset::select('id')->count();

        $issueAssetsCount = [];
        $latestHistory = AssetHealthHistory::orderBy('created_at', 'desc')->first();

        $ignoredTests = AssetHealthTests::where('status', 0)->pluck('id')->toArray();


        if ($latestHistory) {
            $assets = AssetHealthReportDetail::select(DB::raw('count(asset_health_report_details.id) as total'), 'asset_health_reports.category_id', 'asset_health_categories.slug')->join('asset_health_reports', function ($join) use ($latestHistory) {
                $join->on('asset_health_report_details.asset_health_report_id', '=', 'asset_health_reports.id')
                    ->where('asset_health_reports.history_id', '=', $latestHistory->id);
            })
                ->where('asset_health_report_details.is_ignored', 0);

            if (!empty($ignoredTests)) {
                $assets = $assets->whereNotIn('asset_health_reports.test_id', $ignoredTests);
            }

            $assets = $assets->join('asset_health_categories', 'asset_health_reports.category_id', '=', 'asset_health_categories.id')->groupBy('asset_health_reports.category_id')->get();

            $issueAssetsCount = $assets->pluck('total', 'slug')->toArray();
        }

        return ['totalAssetsCount' => $totalAssetsCount, 'issueAssetsCount' => $issueAssetsCount];
    }


    /**
     * Get data for scanned assets including those with no issues and test-wise counts.
     *
     * @return array
     */
    public function getAssetHealthscannedAssetsData()
    {
        $assetsCountData = $this->getIssuesAssetsCount();

        $scannedAssetsData['Asset With No Issues'] = $assetsCountData['totalAssetsCount'] - array_sum($assetsCountData['issueAssetsCount']) ?: $assetsCountData['issueAssetsCount'];

        $latestHistory = AssetHealthHistory::orderBy('created_at', 'desc')->first();

        if ($latestHistory) {
            $testReports = AssetHealthReports::with('assetHealthTest')
                ->where('history_id', $latestHistory->id)
                ->where('test_count', '!=', 0)
                ->get();

            foreach ($testReports as $report) {
                $scannedAssetsData[$report->assetHealthTest->test_name] = $report->test_count;
            }
        }

        return $scannedAssetsData;
    }

    /**
     * Toggle ignoring asset health test detail by its ID.
     *
     * @param  int  $testDetailId
     * @param  bool  $ignoreState
     * @return bool
     */
    public function toggleIgnoreAssetHealthTestDetailById($testDetailId, $ignoreState)
    {
        $assetReportDetail = AssetHealthReportDetail::find($testDetailId);
        $assetReport = AssetHealthReports::find($assetReportDetail->asset_health_report_id);
        $assetHealthHistory = AssetHealthHistory::find($assetReport->history_id);

        $isIgnored = $ignoreState ? 1 : 0;

        $assetReportDetail->update(['is_ignored' => $isIgnored]);
        $adjustment = $ignoreState ? -1 : 1;
        $assetReport->update(['test_count' => $assetReport->test_count + $adjustment]);
        $assetHealthHistory->update(['no_of_issues' => $assetHealthHistory->no_of_issues + $adjustment]);

        return true;
    }

    /**
     * Get all asset health histories within a specified date range.
     *
     * @param  string  $fromDate
     * @param  string  $toDate
     *
     * @return \Illuminate\Support\Collection
     */
    public function getAllAssetHealthHistories($fromDate, $toDate)
    {
        return AssetHealthHistory::select(
            DB::raw('WEEK(created_at, 1) as week'),
            DB::raw('MAX(no_of_issues) as count')
        )
            ->where('asset_health_histories.started_at', '>', $fromDate)
            ->where('asset_health_histories.ended_at', '<', $toDate)
            ->groupBy('week')
            ->get();
    }

    /**
     * Get the latest asset health history entry.
     *
     * @return \App\Models\AssetHealthHistory|null
     */
    public function getLatestAssetHealthHistory()
    {
        return AssetHealthHistory::orderBy('created_at', 'desc')->first();
    }
}
