<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use App\Models\MakeAndModel;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class TechnicalSpecs extends Model
{
    use HasFactory;
    protected $guarded = [];

    public function makeAndModel()
    {
        return $this->belongsTo('App\Models\MakeAndModel');
    }

    public function assetType()
    {
        return $this->belongsTo('App\Models\AssetType');
    }

    public function assets()
    {
        return $this->hasMany('App\Models\Asset', 'technical_spec_id');
    }

    public function scopeAccessories($query)
    {
        return $query->whereIn(
            'make_and_model_id',
            MakeAndModel::accessories()->pluck('id')
        );
    }

    public function scopeGetByAssetType($query, $assetTypeId)
    {
        return $query->whereIn(
            'make_and_model_id',
            MakeAndModel::whereIn('asset_type_id', $assetTypeId)->pluck('id')
        );
    }

    public function scopeGetByManufacturer($query, $manufacturer)
    {
        return $query->whereIn(
            'make_and_model_id',
            MakeAndModel::whereIn('manufacturer_id', $manufacturer)->pluck('id')
        );
    }

    public function scopeGetById($query, $id)
    {
        return $query->whereIn('id', $id);
    }

    public function scopeGetTechnicalSpecs($query, $makeAndModelId)
    {
        return $query->where('make_and_model_id', $makeAndModelId);
    }

    public function scopeGetTechnicalSpecsFromModels($query, $makeAndModelId)
    {
        if (is_array($makeAndModelId)) {
            return $query->whereIn('make_and_model_id', $makeAndModelId);
        }
        return $query->where('make_and_model_id', $makeAndModelId);
    }

    public function scopeGetServer($query)
    {
        return $query->whereIn('make_and_model_id', MakeAndModel::getServer()->pluck('id'))->orderBy('details');
    }

    public function scopeGetNetwork($query)
    {
        return $query->whereIn('make_and_model_id', MakeAndModel::getNetwork()->pluck('id'))->orderBy('details');
    }

    public function scopeGetIT($query)
    {
        return $query->whereIn('make_and_model_id', MakeAndModel::getItModels()->pluck('id'))->orderBy('details');
    }

    public function scopeGetAV($query)
    {
        return $query->whereIn('make_and_model_id', MakeAndModel::getAV()->pluck('id'))->orderBy('details');
    }

    public function scopeGetMobile($query)
    {
        return $query->whereIn('make_and_model_id', MakeAndModel::getMobile()->pluck('id'))->orderBy('details');
    }

    public function scopeActive($query)
    {
        return $query->where('status', 1);
    }

    public function scopeInActive($query)
    {
        return $query->whereNull('status')->orWhere('status', '!=', 1);
    }

    public function scopeGetResearch($query)
    {
        return $query->whereIn('make_and_model_id', MakeAndModel::getResearch()->pluck('id'))->orderBy('details');
    }

    public function scopeGetWithoutAccessories($query)
    {
        return $query->whereIn('make_and_model_id', MakeAndModel::getWithoutAccessories()->pluck('id'))->orderBy('details');
    }

    public function scopeWithoutBlanks($query)
    {
        return $query->whereNotNull('details');
    }


    /**
     * Get the details attribute.
     *
     * @param string $details description
     * @return string
     */
    public function getDetailsAttribute($details)
    {
        return ucfirst($details);
    }

    /**
     * Fetching valid  technical specs for the assets
     *
     * @return object
     */
    public function scopeValid($query)
    {
        $assetTypes = MakeAndModel::whereIn('asset_type_id', AssetType::whereNotIn('slug', ['accessories'])->pluck('id')->toArray())->pluck('id')->toArray();

        return $query->whereIn('make_and_model_id', $assetTypes)
            ->whereNotNull('details');
    }

    /**
     * Get technical specs under specific asset types
     *
     * @param mixed $query
     * @param mixed $assetTypes
     * @return Collection
     */
    public function scopeGetWithAssetTypes($query, array $assetTypes)
    {
        return $query->whereHas('makeAndModel', function ($query) use ($assetTypes) {
            $query->whereIn('asset_type_id', AssetType::whereIn('slug', $assetTypes)->pluck('id')->toArray());

        })->whereNotNull('details');
    }
}
