<?php

namespace App\Models\SoftwareLicense;

use Illuminate\Database\Eloquent\Model;

class SoftwareLicenseUser extends Model
{

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $guarded = [];

    /**
     * Relation with User
     */
    public function user()
    {
        return $this->belongsTo('App\User', 'user_id');
    }

    /**
     * Relation with license
     */
    public function license()
    {
        return $this->belongsTo('App\Models\SoftwareLicense\SoftwareLicense', 'software_license_id');
    }

    /**
     * Relation with license key
     */
    public function licenseKey()
    {
        return $this->belongsTo('App\Models\SoftwareLicense\SoftwareLicenseKey', 'software_license_key_id');
    }

    /**
     * format created at date
     */
    public function getCreatedAtAttribute($date)
    {
        return parse_date_from_db_datetime($this->attributes['created_at']);
    }

    /**
     * Scope a query to only include users who have a software license and are not Dropbox members.
     **/
    public function scopeNotDropboxMembers($query, $dropboxSoftwareId)
    {
        return $query->whereNotIn('user_id', function ($subQuery) use ($dropboxSoftwareId) {
            $subQuery->select('user_id')
                ->from('dropbox_members')
                ->whereNotNull('user_id');
        })
            ->where('software_license_id', $dropboxSoftwareId);
    }

    /**
     * Scope a query to only include users who have a Zoom software license and they are not Zoom members as per API now.
     * 
     * @param object $query
     * @param int $zoomSoftwareId
     * 
     * @return object
     **/
    public function scopeNotZoomMembers($query, $zoomSoftwareId)
    {
        return $query->whereNotIn('user_id', function ($subQuery) {
            $subQuery->select('user_id')
                ->from('zoom_members')
                ->whereNotNull('user_id');
        })
            ->where('software_license_id', $zoomSoftwareId);
    }

    /**
     * Scope a query to only include users who have a Slack software license and they are not Zoom members as per API now.
     * 
     * @param object $query
     * @param int $slackSoftwareId
     * 
     * @return object
     **/
    public function scopeNotSlackMembers($query, $slackSoftwareId)
    {
        return $query->whereNotIn('user_id', function ($subQuery) {
            $subQuery->select('user_id')
                ->from('slack_members')
                ->whereNotNull('user_id');
        })
            ->where('software_license_id', $slackSoftwareId);
    }

    /**
     * Retrieves the related ZoomMember for this model's user.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function zoomMember()
    {
        return $this->belongsTo(\App\Models\ZoomMember::class, 'user_id', 'user_id');
    }
}
