<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use App\Models\TechnicalSpecs;
use App\Models\Accessory;
use App\Models\Manufacturer;
use App\Models\AssetType;

class MakeAndModel extends Model
{
	protected $guarded = [];

	public function create($name, $manufacturer_id)
	{
		return $this->firstOrCreate([
    		'manufacturer_id' => $manufacturer_id,
    		'name' => $name
    	], [
    		'slug' => str_slug($name, '_'),
    	]);
	}

	public function getAll()
	{
		return MakeAndModel::with(['manufacturer' => function ($query) {
            $query->orderBy('manufacturers.slug');
        }])->get();
	}

    public function scopeGetOrderByName($query)
    {
        return $query->with('manufacturer')->get()
            ->sortBy(function($makeAndModel) {
                return $makeAndModel->manufacturer->slug;
            });
    }

	public function technicalSpecs()
	{
		return $this->hasMany(TechnicalSpecs::class);
	}

	public function manufacturer()
	{
		return $this->belongsTo('App\Models\Manufacturer')->orderBy('manufacturers.slug');
	}

    public function assetType()
    {
    	return $this->belongsTo('App\Models\AssetType')->orderBy('asset_types.slug');
    }
	public function scopeAccessories($query)
	{
		return $query->where('asset_type_id',
			AssetType::getBySlug('accessories')->first()->id
		);
	}

	public static function scopeManufacturer($name)
	{
		return Manufacturer::where('slug','like', str_slug($name))->pluck('id');

	}

	public function scopeGetMakeModel($query, $assetTypeId, $manufacturerId)
    {
        return $query->where('asset_type_id', $assetTypeId)
                    ->where('manufacturer_id', $manufacturerId);
    }

    public function scopeGetMakeModelFromAssetTypes($query, $assetTypeId)
    {
        if(is_array($assetTypeId)) {
            return $query->whereIn('asset_type_id', $assetTypeId);
        }
        return $query->where('asset_type_id', $assetTypeId);
    }

    public function scopeGetServer($query)
    {
    	  return $query->whereIn('asset_type_id', AssetType::whereIn('slug',['servers', 'server_rack','server_components'])->pluck('id'));
    }

    public function scopeGetItModels($query)
    {
        return $query->whereHas('assetType', function ($query) {
            $query->itAsset();
        });
    }

    public function scopeGetMobile($query)
    {
        return $query->whereHas('assetType', function ($query) {
            $query->mobileAsset();
        });
    }

    public function scopeGetAV($query)
    {
        return $query->whereHas('assetType', function ($query) {
            $query->avAsset();
        });
    }

    public function scopeGetNetwork($query)
    {
        return $query->whereHas('assetType', function ($query) {
            $query->networkAsset();
        });
    }

    public function getMakeModelNameAttribute()
    {
        return is_null($this->manufacturer) ? '' : ucfirst($this->manufacturer->attributes['name']).' - '.$this->attributes['name'];
    }

    public function scopeGetResearch($query)
    {
        return $query->whereHas('assetType', function ($query) {
            $query->researchAsset();
        });
    }

    public function assets()
    {
        return $this->hasMany('App\Models\Asset', 'make_and_model_id');
    }

    public function scopeGetWithoutAccessories($query)
    {
          return $query->where('asset_type_id', '!=', AssetType::where('slug','accessories')->first()->id);
    }

    /**
     * Scope to get records without specific asset types.
     *
     * @param mixed $query
     * @param mixed $assetTypes
     * @return mixed
     */
    public function scopeGetWithoutAssetTypes($query, $assetTypes)
    {
        if (is_array($assetTypes)) {
            return $query->whereIn('asset_type_id', AssetType::whereNotIn('slug', $assetTypes)->pluck('id')->toArray());
        }

        return $query->where('asset_type_id', '!=', AssetType::where('slug', '=', $assetTypes)->first()->id);
    }

    /**
     * Scope to get records with specific asset types.
     *
     * @param mixed $query
     * @param mixed $assetTypes
     * @return mixed
     */
    public function scopeGetWithAssetTypes($query, $assetTypes)
    {
        if (is_array($assetTypes)) {
            return $query->whereIn('asset_type_id', AssetType::whereIn('slug', $assetTypes)->pluck('id')->toArray());
        }

        return $query->where('asset_type_id', AssetType::where('slug', '=', $assetTypes)->first()->id);
    }


    /**
     * Scope to get records with specific asset type ids.
     *
     * @param mixed $query
     * @param mixed $assetTypeIds
     * @return mixed
     */
    public function scopeGetWithAssetTypeIds($query, $assetTypeIds)
    {
        if (is_array($assetTypeIds)) {
            return $query->whereIn('asset_type_id', $assetTypeIds);
        }

        return $query->where('asset_type_id', $assetTypeIds);
    }


    /**
     * Fetching valid asset types for the assets
     *
     * @return object
     */
    public function scopeValid($query)
    {
        return $query->whereIn('asset_type_id', AssetType::whereNotIn('slug', ['accessories'])->pluck('id')->toArray());
    }
}
