<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use App\Models\LocationType;
use App\Models\AssetType;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Location extends Model
{
    protected $guarded = [];
    use HasFactory;

    public function assets()
    {
        return $this->hasMany('App\Models\Asset');
    }

    public function accessories()
    {
        return $this->hasMany('App\Models\Accessory');
    }

    public function state()
    {
        return $this->belongsTo('App\Models\State');
    }

    public function locationType()
    {
        return $this->belongsTo('App\Models\LocationType');
    }


    public function region()
    {
        return $this->belongsTo('App\Models\Region');
    }

    public function country()
    {
        return $this->belongsTo('App\Models\Country');
    }

    public function siteCode()
    {
        return $this->belongsTo('App\Models\SiteCode');
    }

    public function scopeWithoutwarehouse($query)
    {
        return $query->whereNotIn('location_type_id', LocationType::where('location_type_name', 'Storage')->get()->pluck('id'));
    }

    public function scopeWarehouses($query)
    {

        return $query->whereIn('location_type_id', LocationType::where('location_type_name', 'Storage')->get()->pluck('id'));
    }

    public function scopeInstallBuildings($query)
    {
        return $query->whereIn('location_type_id', LocationType::where('location_type_name', 'Install')->orWhere('location_type_name', 'Building')->get()->pluck('id'));
    }

    public function scopeInstallLocations($query)
    {
        return $query->whereIn('location_type_id', LocationType::where('location_type_name', 'Install')->get()->pluck('id'));
    }

    public function scopeServerLocation($query)
    {

        return $query->whereIn('location_type_id', LocationType::whereIn('location_type_name', ['Warehouse', 'Server Room'])->get()->pluck('id'));
    }


    public function scopeRegularLocations($query)
    {
        return $query->where('room_name', '!=', 'OUT OF UBER');
    }

    public function mobile_phone()
    {
        return $this->assets()->mobileAsset();
    }

    public function network()
    {
        return $this->assets()->networkAsset();
    }

    public function av()
    {
        return $this->assets()->avAsset();
    }

    public function research()
    {
        return $this->assets()->researchAsset();
    }

    public function scopeGetFromName($query, $name)
    {
        return $query->where('room_name', $name);
    }

    public function scopeActive($query)
    {
        return $query->where('status', 1);
    }

    public function scopeInactive($query)
    {
        return $query->where('status', 0);
    }

    public function scopeLocationWhere($query, $searchText)
    {
        return $query->where('room_name', 'like', '%' . $searchText . '%');
    }

    public function scopeVc($query)
    {
        return $query->where('vc', 'VC');
    }

    public function scopeNotVc($query)
    {
        return $query->where('vc', NULL)->orWhere('vc', '');
    }

    public function scopeValid($query)
    {
        return $query->where('room_name', '!=', 'DO NOT USE LOCATION - Initial Load Anup Only');
    }

    public function scopeFreezed($query)
    {
        return $query->where('is_freeze', 1);
    }

    public function scopeUnfreezed($query)
    {
        return $query->where('is_freeze', 0);
    }

    public function department()
    {
        return $this->belongsTo('App\Models\Department');
    }

    /**
     * Scope a query to only include locations where the zoom value is 'Zoom'.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query The query builder instance.
     *
     * @return \Illuminate\Database\Eloquent\Builder The modified query builder instance with the zoom filter applied.
     */
    public function scopeZoom($query)
    {
        return $query->where('zoom', 'Zoom');
    }

    /**
     * Scope a query to only include locations where the zoom value is null or empty string.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query The query builder instance.
     *
     * @return \Illuminate\Database\Eloquent\Builder The modified query builder instance with the zoom filter applied.
     */
    public function scopeNotZoom($query)
    {
        return $query->where('zoom', null)->orWhere('zoom', '');
    }

    /**
     * Get the Zoom devices associated with the location.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function zoomDevices()
    {
        return $this->hasMany('App\Models\ZoomDevice');
    }
}
