<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Cache;

class AssetType extends Model
{
    protected $guarded = [];

    public function assets()
    {
        return $this->hasMany('App\Models\Asset');
    }

    public function makeAndModel()
    {
        return $this->hasMany('App\Models\makeAndModel');
    }

    public function assetTabs()
    {
        return $this->belongsToMany('App\Models\AssetTab');
    }

    public function hasRelationTo($assetTabs)
    {
        return $this->assetTabs->contains('id', $assetTabs->id);
    }

    public function hasRelationToSlug($slug)
    {
        return $this->assetTabs->contains('slug', $slug);
    }

    public function scopeItAsset($query)
    {
        return $query->whereHas('assetTabs', function ($query) {
            $query->where('slug', 'it_assets');
        });
    }

    public function scopeAvAsset($query)
    {
        return $query->whereHas('assetTabs', function ($query) {
            $query->where('slug', 'av_asset');
        });
    }

    public function scopeMobileAsset($query)
    {
        return $query->whereHas('assetTabs', function ($query) {
            $query->where('slug', 'mobile_assets');
        });
    }

    public function scopeNetworkAsset($query)
    {
        return $query->whereHas('assetTabs', function ($query) {
            $query->where('slug', 'network_asset');
        });
    }

    public function scopeResearchAsset($query)
    {
        return $query->whereHas('assetTabs', function ($query) {
            $query->where('slug', 'research_asset');
        });
    }

    public function scopeGetBySlug($query, $slug)
    {
        if (is_array($slug)) {
            return $query->whereIn('slug', $slug);
        }
        return $query->where('slug', $slug);
    }

    public function scopeRegular()
    {
        return Cache::remember('regular-asset-types', 1440, function () {
            return $this->itAsset()->orderBy('slug', 'ASC')->get();
        });
    }

    public function scopeMobile()
    {
        return $this->mobileAsset()->orderBy('slug', 'ASC')->get();
    }

    public function scopeNetwork()
    {
        return $this->networkAsset()->orderBy('slug', 'ASC')->get();
    }

    public function scopeAvassets()
    {
        return $this->avAsset()->orderBy('slug', 'ASC')->get();
    }

    public function scopeResearch()
    {
        return $this->researchAsset()->orderBy('slug', 'ASC')->get();
    }

    public static function server()
    {
        return AssetType::whereIn('slug', ['servers', 'server_rack', 'server_components'])->orderBy('slug', 'ASC')->get();
    }

    public static function allWithoutComputerAccessories()
    {
        return AssetType::where('slug', '!=', 'computer_accessories')->orderBy('slug', 'ASC')->get();
    }

    public function scopeWithoutSlug($query, $slug)
    {
        return $query->whereNotIn('slug', $slug)->orderBy('slug', 'ASC')->get();
    }

    public function scopeLeased($query)
    {
        return $query->where('leased', 1);
    }

    public function scopeGetFromName($query, $name)
    {
        return $query->where('name', $name);
    }

    /**
     * Scope for getting assets type list those are excluded for variance report.
     *
     * @param mixed $query
     *
     * @return mixed $query
     */
    public function scopeGetAssetTypesExcludedInVarianceReport($query)
    {
        $excludedAssetTypes = config('cycle-count.excluded-asset-types', []);
        if (empty($excludedAssetTypes)) {
            return $query;
        }
        return $query->whereIn('slug', $excludedAssetTypes);
    }

    /**
     * Retrieves records from the database based on a list of slugs.
     *
     * @param mixed $query The database query builder instance.
     * @param mixed $slug An array of slugs to match against.
     * @return mixed The modified query builder instance.
     */
    public function scopeGetBySlugList(mixed $query, mixed $slug = [])
    {
        return $query->whereIn('slug', $slug);
    }

    /**
     * Scope to get records without specific asset types.
     *
     * @param mixed $query
     * @param mixed $assetTypes
     * @return mixed
     */
    public function scopeGetWithoutAssetTypes($query, $assetTypes)
    {
        if (is_array($assetTypes)) {
            return $query->whereNotIn('slug', $assetTypes);
        }

        return $query->where('slug', '!=', $assetTypes);

    }

    /**
     * Fetching valid asset types for the assets
     *
     * @return object
     */
    public function scopeValid($query)
    {
        return $query->whereNotIn('slug', ['accessories']);
    }
}
