<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class AssetStatus extends Model
{
    protected $guarded = [];

    public function assets()
    {
        return $this->hasMany('App\Models\Asset');
    }

    public function scopeId($query, $name)
    {
        return $query->where('name', $name);
    }

    public function scopeGetName($query, $id)
    {
        return $query->where('id', $id);
    }

    public function scopeGetNameFromSlug($query, $slug)
    {
        if (is_array($slug)) {
            return $query->whereIn('slug', $slug);
        }
        return $query->where('slug', $slug);
    }

    public function scopeGetSlugFromName($query, $name)
    {
        return $query->where('name', $name);
    }

    public function scopeReturn($query)
    {
        return $query->where('can_be_returned', 1);
    }

    public function scopeGetReturnStatus($query)
    {
        return $query->whereNotIn('slug', ['stolen_lost', 'end_of_life_data_wiped', 'assigned', 'brand_new', 'installed', 'loaned'])->orderby('slug', 'ASC');
    }

    public function scopeGetStatusID($query, $slug)
    {
        if (is_array($slug)) {
            return $query->whereIn('slug', $slug);
        }
        return $query->where('slug', $slug);
    }

    public function scopeOnHandStatuses($query)
    {
        return $query->where('has_location', 1);
    }


    /**
     * Scope for getting assets status list those are excluded for variance report.
     *
     * @param mixed $query
     *
     * @return mixed $query
     */
    public function scopeGetAssetStatusesExcludedInVarianceReport($query)
    {
        $excludedAssetStatuses = config('cycle-count.excluded_asset_statuses', []);
        if (empty($excludedAssetStatuses)) {
            return $query;
        }
        return $query->whereIn('slug', $excludedAssetStatuses);
    }

    public function scopeSkipStatusesForComputerReport($query)
    {
        $skipArray = ['cdw_in_transit', 'connection_in_transit', 'end_of_life_data_wiped', 'end_of_life', 'stolen_lost', 'rma_returned'];

        return $query->whereNotIn('slug', $skipArray);
    }

    public function scopeGetInventoryAssetStatus($query)
    {
        return $query->whereIn('slug', config('reports.consumption_report.include_status'));
    }

    public function scopeWithUser($query)
    {
        return $query->where('has_user', 1)->orderby('slug', 'ASC');
    }

    public function scopeListStatus($query)
    {
        return $query->where('slug', '!=', 'cdw_in_transit');
    }

    /**
     * Filters the query to exclude assets with specified statuses.
     *
     * @param mixed $query The query to filter.
     * @param array $assetsStatus The statuses to exclude.
     * @return mixed The filtered query.
     */
    public function scopeGetWithoutAssetStatus($query, array $assetsStatus)
    {
        if (!$assetsStatus) {
            return $query;
        }

        return $query->whereNotIn('slug', $assetsStatus);
    }

    /**
     * Scope a query to only include records with specific asset statuses.
     *
     * @param array $assetsStatus
     * @return mixed
     */
    public function scopeGetWithAssetStatus($query, array $assetsStatus)
    {
        if (!$assetsStatus) {
            return $query;
        }

        return $query->whereIn('slug', $assetsStatus);
    }

    /**
     * Scope a query to only include valid items.
     */
    public function scopeReportStatus($query)
    {
        return $query->whereNotIn('slug', config('reports.common_reports.exclude_statuses'));
    }

    /**
     * Scope a query to include only managed devices with statuses indicating they are deployed but not checked in.
     *
     * This scope filters the query to include only managed devices with statuses that typically indicate
     * they are deployed but have not been checked in, such as "assigned" or "loaned".
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeManagedDeployedNotCheckedInStatuses($query)
    {
        return $query->whereIn('slug', ['assigned', 'loaned']);
    }

    /**
     * Scope a query to include only not managed, deployed, not checked in statuses.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeNotManagedDeployedNotCheckedInStatuses($query)
    {
        return $query->whereIn('slug', ['assigned', 'loaned']);
    }

    /**
     * Scope a query to include only devices with statuses indicating they are not currently active.
     *
     * This scope filters the query to include only devices with statuses that typically indicate
     * they are not currently active, such as "assigned" or "loaned".
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeAssignedDevicesNotActiveStatuses($query)
    {
        return $query->whereIn('slug', ['assigned', 'loaned']);
    }
}
