<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\DB;

class Asset extends Model
{
    protected $guarded = [];

    use HasFactory;

    public function location()
    {
        return $this->belongsTo('App\Models\Location');
    }

    public function user()
    {
        return $this->belongsTo('App\User');
    }

    public function assetHistory()
    {
        return $this->hasMany('App\Models\AssetHistory')->orderByDesc('id');
    }

    public function airwatch()
    {
        return $this->hasOne('App\Models\AirWatch');
    }

    public function chromebook()
    {
        return $this->hasOne('App\Models\Chromebook');
    }

    public function assetTypeAttributeValue()
    {
        return $this->hasMany('App\Models\AssetTypeAttributeValue');
    }

    public function jamf()
    {
        return $this->hasOne('App\Models\JamfComputer');
    }

    public function intune()
    {
        return $this->hasOne('App\Models\IntuneComputer');
    }

    public function kandjiDevices()
    {
        return $this->hasOne('App\Models\KandjiDevices');
    }

    public function mobileIron()
    {
        return $this->hasOne('App\Models\MobileIron');
    }

    public function latestAssetHistory()
    {
        return $this->hasOne('App\Models\AssetHistory')->latest();
    }

    public function scopeStatusEmpty($query)
    {
        return $query->where(function ($query) {
            return $query->whereNull('asset_status_id')
                ->orWhere('asset_status_id', 0);
        });
    }


    public function scopeUserNotAssigned($query)
    {
        return $query->where(function ($query) {
            return $query->whereNull('user_id')
                ->orWhere('user_id', 0);
        });
    }

    public function scopeUserAssigned($query)
    {
        return $query->where('user_id', '>', 0);
    }

    public function scopeRegularAsset($query)
    {
        return $query->whereHas('assetType', function ($query) {
            $query->itAsset();
        });
    }

    public function scopeAvAsset($query)
    {
        return $query->whereHas('assetType', function ($query) {
            $query->avAsset();
        });
    }

    public function scopeMobileAsset($query)
    {
        return $query->whereHas('assetType', function ($query) {
            $query->mobileAsset();
        });
    }

    public function scopeNetworkAsset($query)
    {
        return $query->whereHas('assetType', function ($query) {
            $query->networkAsset();
        });
    }

    public function scopeResearchAsset($query)
    {
        return $query->whereHas('assetType', function ($query) {
            $query->researchAsset();
        });
    }

    public function scopeServerAsset($query)
    {
        return $query->whereIn('assets.asset_type_id', AssetType::whereIn('slug', ['servers', 'server_components', 'server_rack'])->pluck('id'));
    }

    public function scopeGetByTag($query, $tag)
    {
        return $query->where('asset_tag', $tag);
    }

    public function scopeGetBySerial($query, $serial)
    {
        return $query->where('serial_no', $serial);
    }

    public function scopeGetByTagAndSerial($query, $tag)
    {
        return $query->where('serial_no', $tag)->orWhere('asset_tag', $tag);
    }

    public function assetType()
    {
        return $this->belongsTo('App\Models\AssetType');
    }

    public function makeAndModel()
    {
        return $this->belongsTo('App\Models\MakeAndModel');
    }

    public function technicalSpec()
    {
        return $this->belongsTo('App\Models\TechnicalSpecs');
    }

    public function assetStatus()
    {
        return $this->belongsTo('App\Models\AssetStatus');
    }

    /**
     * Get the lock status for this asset.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function lockStatus()
    {
        return $this->belongsTo('App\Models\LockStatus');
    }

    public function canBeMoved()
    {
        if ($this->parent_asset_id) {
            return 0;
        }
        if ($this->assetStatus->slug != 'installed') {
            return $this->assetStatus->has_location;
        }
        return 0;
    }


    public function scopeHasAssetType($query, $slug)
    {
        return $query->whereHas('assetType', function ($query) use ($slug) {
            if (is_array($slug)) {
                return $query->whereIn('slug', $slug);
            }
            return $query->where('slug', $slug);
        });
    }


    public function scopeHasAnyAssetType($query, array $slugs)
    {
        return $query->whereHas('assetType', function ($query) use ($slugs) {
            $query->whereIn('slug', $slugs);
        });
    }

    public function scopeHasAnyAssetStatus($query, array $slugs)
    {
        return $query->whereHas('assetStatus', function ($query) use ($slugs) {
            $query->whereIn('slug', $slugs);
        });
    }

    public function canBeReturned()
    {
        if ($this->parent_asset_id) {
            return 0;
        }
        return $this->assetStatus->has_user && !$this->assetStatus->has_location;
    }

    public function canBeLinked()
    {
        return $this->assetStatus->can_be_linked;
    }

    public function parentAsset()
    {
        return $this->hasOne('App\Models\Asset', 'id', 'parent_asset_id');
    }

    public function childrenAsset()
    {
        return $this->hasMany('App\Models\Asset', 'parent_asset_id', 'id');
    }

    public function carrier()
    {
        return $this->belongsTo('App\Models\Carrier');
    }

    public function setDateDeployedAttribute($date)
    {
        return $this->attributes['date_deployed'] =  convert_to_db_date($date);
    }

    public function getDateDeployedAttribute($date)
    {
        return parse_date_from_db_date($date);
    }

    public function setLeaseStartDateAttribute($date)
    {
        return $this->attributes['lease_start_date'] =  convert_to_db_date($date);
    }

    public function getLeaseStartDateAttribute($date)
    {
        return parse_date_from_db_date($date);
    }

    public function setLeaseEndDateAttribute($date)
    {
        return $this->attributes['lease_end_date'] =  !empty($date) ? convert_to_db_date($date) : null;
    }

    public function getLeaseEndDateAttribute($date)
    {
        return parse_date_from_db_date($date);
    }

    public function setWarrantyEndDateAttribute($date)
    {
        return $this->attributes['warranty_end_date'] = !empty($date) ? convert_to_db_date($date) : null;
    }

    public function setCertificateAddedDateAttribute($date)
    {
        return $this->attributes['certificate_added_date'] =  !empty($date) ? convert_to_db_date($date) : null;
    }

    public function getCertificateAddedDateAttribute($date)
    {
        return parse_date_from_db_date($date);
    }

    public function setLoanerReturnDateAttribute($date)
    {
        return $this->attributes['loaner_return_date'] =  convert_to_db_date($date);
    }

    public function getLoanerReturnDateAttribute($date)
    {
        return parse_date_from_db_date($date);
    }

    public function setLostDateAttribute($date)
    {
        return $this->attributes['lost_date'] =  convert_to_db_date($date);
    }

    public function getLostDateAttribute($date)
    {
        return parse_date_from_db_date($date);
    }

    public function setSerialNoAttribute($serialNo)
    {
        if (starts_with(lcfirst($serialNo), 's')) {
            $serialNo = ltrim($serialNo, 's');
        }

        return $this->attributes['serial_no'] = $serialNo;
    }

    public function setEndOfLifeDateAttribute($date)
    {
        return $this->attributes['end_of_life_date'] =  convert_to_db_date($date);
    }

    public function getEndOfLifeDateAttribute($date)
    {
        return parse_date_from_db_date($date);
    }

    public function setLoanerRetentionDateAttribute($date)
    {
        return $this->attributes['loaner_retention_date'] =  convert_to_db_date($date);
    }

    public function getLoanerRetentionDateAttribute($date)
    {
        return parse_date_from_db_date($date);
    }

    /**
     * Set the return_hold_wipe_confirmed_date attribute
     *
     * @param string $date Date to be set
     *
     * @return void
     */
    public function setRetainHoldWipeConfirmedDateAttribute($date)
    {
        $this->attributes['retain_hold_wipe_confirmed_date'] = convert_to_db_date($date);
    }

    /**
     * Return the retain hold wipe confirmation date in user format
     *
     * @return string
     */
    public function getRetainHoldWipeConfirmationDateAttribute()
    {
        if (array_key_exists('retain_hold_wipe_confirmed_date', $this->attributes)) {
            return parse_date_from_db_date($this->attributes['retain_hold_wipe_confirmed_date']);
        }

        return null;
    }

    // created for asset edit view
    public function getCreatedAtAttribute()
    {
        return parse_date_from_db_datetime($this->attributes['created_at']);
    }

    public function getAssetAgeAttribute()
    {
        $varDate = Carbon::createFromFormat('m/d/Y', parse_date_from_db_datetime($this->attributes['created_at']));

        return age($varDate);
    }

    public function getLinkedDateAttribute($date)
    {
        return parse_date_from_db_date($date);
    }

    public function scopeParent($query)
    {
        return $query->whereHas('assetType', function ($query) {
            $query->where('parent', 1);
            // ->whereNotIn('slug', ['mobile_phone', 'mobile_phone_number']);
        });
    }

    public function scopeParentServer($query)
    {
        return $query->whereHas('assetType', function ($query) {
            $query->where('parent', 1)
                ->whereIn('slug', ['servers', 'server_rack', 'server_components']);
        });
    }

    public function scopeParentMobile($query)
    {
        return $query->whereHas('assetType', function ($query) {
            $query->where('parent', 1)
                ->whereIn('slug', ['mobile_phone', 'mobile_phone_number']);
        });
    }

    public function scopeChild($query)
    {
        return $query->whereHas('assetType', function ($query) {
            $query->where('parent', '!=', 1);
            // ->whereNotIn('slug', ['mobile_phone', 'mobile_phone_number']);
        });
    }

    public function scopeChildServer($query)
    {
        return $query->whereHas('assetType', function ($query) {
            $query->where('parent', '!=', 1)
                ->whereIn('slug', ['servers', 'server_rack', 'server_components']);
        });
    }

    public function scopeChildMobile($query)
    {
        return $query->whereHas('assetType', function ($query) {
            $query->where('parent', '!=', 1)
                ->whereIn('slug', ['mobile_phone', 'mobile_phone_number']);
        });
    }

    /**
     * Get the depreciated value attribute.
     *
     * @return float
     */
    public function getDepreciatedValueAttribute()
    {
        $originalValue = is_numeric($this->original_value) ? (float) $this->original_value : 0;

        $varDate = Carbon::createFromFormat('m/d/Y', parse_date_from_db_datetime($this->attributes['created_at']));
        $years = Carbon::now()->diff($varDate)->format('%y');

        if ($years > 0) {
            return round($originalValue - (($originalValue * 20) / 100) * $years, 2);
        }

        return $originalValue;
    }

    public function scopeAssetypeValidateIt($query, $asset)
    {
        return $query->where('serial_no', $asset)->orWhere('asset_tag', $asset)->regularAsset()->get()->count();
    }

    public function scopeAssetypeValidateAvasset($query, $asset)
    {
        return $query->where('serial_no', $asset)->orWhere('asset_tag', $asset)->avAsset()->get()->count();
    }

    public function scopeAssetypeValidateMobile($query, $asset)
    {
        return $query->where('serial_no', $asset)->orWhere('asset_tag', $asset)->mobileAsset()->get()->count();
    }

    public function scopeAssetypeValidateNetwork($query, $asset)
    {
        return $query->where('serial_no', $asset)->orWhere('asset_tag', $asset)->networkAsset()->get()->count();
    }

    public function scopeAssetypeValidateServer($query, $asset)
    {
        return $query->where('asset_tag', $asset)->whereIn('asset_type_id', AssetType::whereIn('slug', ['servers', 'server_rack', 'server_components'])->get()->pluck('id'))->get()->count();
    }

    public function scopeAssetypeValidateResearch($query, $asset)
    {
        return $query->where('serial_no', $asset)->orWhere('asset_tag', $asset)->researchAsset()->get()->count();
    }

    public function scopeInActiveUsers($query)
    {
        return $query->whereHas('user', function ($query) {
            $query->whereNull('status')->orWhere('status', '!=', 1);
        });
    }

    public function scopeInActiveLocations($query)
    {
        return $query->whereHas('location', function ($query) {
            $query->where('status', 0);
        });
    }

    public function scopeHasStatus($query, $slug)
    {
        return $query->whereHas('assetStatus', function ($query) use ($slug) {
            $query->where('slug', $slug);
        });
    }

    public function scopeLostStolenAssetWithoutDataWipe($query)
    {
        return $query->whereHas('assetStatus', function ($query) {
            $query->where('slug', 'stolen_lost');
        })
            ->whereNull('wipe_confirmation');
    }

    public function scopeEndOfLifeWithoutCertificate($query)
    {
        return $query->whereHas('assetStatus', function ($query) {
            $query->where('slug', 'end_of_life_data_wiped');
        })
            ->whereNull('ewaste_certificate');
    }

    public function scopeNotEndofLifeOrDisposed($query)
    {
        return $query->whereHas('assetStatus', function ($query) {
            $query->whereNotIn('slug', ['end_of_life_disposed', 'end_of_life_data_wiped'])
                ->where('slug', '!=', 'end_of_life');
        });
    }

    public function scopeAssetDueWeek($query)
    {
        return $query->hasStatus('loaned')
            ->whereDate('loaner_return_date', '>=', Carbon::now()->startOfWeek()->toDateString())
            ->whereDate('loaner_return_date', '<=', Carbon::now()->endOfWeek()->toDateString());
    }

    public function getLoanerAgeAttribute()
    {
        $varDate = Carbon::createFromFormat('m/d/Y', parse_date_from_db_date($this->attributes['loaner_return_date']));
        return age($varDate);
    }

    public function getLoanerRetentionAgeAttribute()
    {
        $varDate = Carbon::createFromFormat('m/d/Y', parse_date_from_db_date($this->attributes['loaner_retention_date']));
        return age($varDate);
    }

    public function scopeNotSearched($query, $searchText)
    {
        $searchTerms = explode(' ', $searchText);
        return  $query->where(function ($query) use ($searchTerms) {
            foreach ($searchTerms as $searchText) {
                $query->where('asset_tag', '!=', "$searchText")
                    ->where('serial_no', '!=', "$searchText");
            }
        });
    }

    public function scopeRelationData($query, $value, $field)
    {
        if ($field == 'building') {
            if ($value) {
                return $query->whereHas('location', function ($query) use ($value) {
                    if (is_array($value)) {
                        $query->whereIn('building', $value);
                    } else {
                        $query->where('building', $value);
                    }
                });
            }
            return $query;
        }
        if ($value) {
            if (is_array($value)) {
                return $query->whereIn($field, $value);
            }
            return $query->where($field, $value);
        }
        return $query;
    }

    public function scopeAgeOverFourYears($query)
    {
        return $query->whereDate('created_at', '<', Carbon::now()->subYears(4)->toDateString());
    }

    public function scopeDamagedOverThirtyDays($query)
    {
        return $query->hasStatus('damaged')->whereDate('last_status_update', '<', Carbon::now()->subDays(30)->toDateString());
    }

    public function scopeIsMobile($query)
    {
        return $query->whereHas('assetType', function ($query) {
            $query->where('slug', 'mobile_phone');
        });
    }

    /**
     * get last seen
     */
    public function getLastSeenAttribute()
    {
        if ($this->intune) {
            return optional($this->intune)->checkin_date;
        }
        if ($this->jamf) {
            return optional($this->jamf)->checkin_date;
        }
        if ($this->airwatch) {
            return optional($this->airwatch)->last_seen;
        }
        if ($this->chromebook) {
            return optional($this->chromebook)->last_sync;
        }
        if ($this->kandjiDevices) {
            return optional($this->kandjiDevices)->last_checkin;
        }
        if ($this->mobileIron) {
            return optional($this->mobileIron)->last_checkin;
        }

        if ($this->jumpCloud) {
            return optional($this->jumpCloud)->last_seen;
        }

        return '';
    }


    /**
     * get end point last logged user
     */
    public function getEndPointLastLoggedUserAttribute()
    {
        if ($this->intune) {
            return optional($this->intune)->username ?? optional($this->intune)->user_email;
        }
        if ($this->jamf) {
            return optional($this->jamf)->username ?? optional($this->jamf)->jamf_email;
        }
        if ($this->airwatch) {
            return optional($this->airwatch)->user_name ?? optional($this->airwatch)->email;
        }
        if ($this->chromebook) {
            return optional($this->chromebook)->email;
        }
        if ($this->kandjiDevices) {
            return optional($this->kandjiDevices)->user_name ?? optional($this->kandjiDevices)->email;
        }
        if ($this->mobileIron) {
            return optional($this->mobileIron)->first_name . ' ' . optional($this->mobileIron)->last_name;
        }

        if ($this->jumpCloud) {
            return optional($this->jumpCloud)->user_name ?? optional($this->jumpCloud)->email;
        }

        return '';
    }

    /**
     * get hardware standard from discovery tools
     */
    public function getHardwareStandardAttribute()
    {
        if ($this->intune) {
            return optional($this->intune)->model;
        }
        if ($this->jamf) {
            return optional($this->jamf)->model;
        }
        if ($this->airwatch) {
            return optional($this->airwatch)->model;
        }
        if ($this->chromebook) {
            return optional($this->chromebook)->model;
        }
        if ($this->kandjiDevices) {
            return optional($this->kandjiDevices)->model;
        }
        if ($this->mobileIron) {
            return optional($this->mobileIron)->model;
        }

        if ($this->jumpCloud) {
            return optional($this->jumpCloud)->model;
        }

        return '';
    }

    /**
     * get User from discovery tools
     */
    public function getDiscoveryUserAttribute()
    {
        if ($this->intune) {
            return optional($this->intune)->user_name;
        }
        if ($this->jamf) {
            return optional($this->jamf)->jamf_email;
        }
        if ($this->airwatch) {
            return optional($this->airwatch)->first_name . ' ' . optional($this->airwatch)->last_name;
        }
        if ($this->chromebook) {
            return optional($this->chromebook)->email;
        }
        if ($this->kandjiDevices) {
            return optional($this->kandjiDevices)->user_name ?? optional($this->kandjiDevices)->email;
        }
        if ($this->mobileIron) {
            return optional($this->mobileIron)->first_name . ' ' . optional($this->mobileIron)->last_name;
        }

        if ($this->jumpCloud) {
            return optional($this->jumpCloud)->email;
        }

        return '';
    }

    public function assetTracking()
    {
        return $this->belongsTo('App\Models\AssetTracking');
    }

    public function assetComment()
    {
        return $this->hasMany('App\Models\AssetComment');
    }

    public function getEndPointHostNameAttribute()
    {
        if ($this->airwatch) {
            return optional($this->airwatch)->host_name;
        }
        if ($this->chromebook) {
            return optional($this->chromebook)->host_name;
        }

        return '';
    }

    public function getEndPointMacAddressAttribute()
    {
        if ($this->intune) {
            return optional($this->intune)->mac_address;
        }
        if ($this->jamf) {
            return optional($this->jamf)->mac_address;
        }
        if ($this->airwatch) {
            return optional($this->airwatch)->mac_address;
        }
        if ($this->chromebook) {
            return optional($this->chromebook)->mac_address;
        }
        if ($this->kandjiDevices) {
            return '';
        }
        if ($this->mobileIron) {
            return optional($this->mobileIron)->mac_address;
        }

        return '';
    }

    public function getEndPointPlatformAttribute()
    {
        if ($this->intune) {
            return optional($this->intune)->platform;
        }
        if ($this->airwatch) {
            return optional($this->airwatch)->platform;
        }
        if ($this->chromebook) {
            return optional($this->chromebook)->platform;
        }
        if ($this->kandjiDevices) {
            return optional($this->kandjiDevices)->platform;
        }
        if ($this->mobileIron) {
            return optional($this->mobileIron)->platform;
        }

        if ($this->jumpCloud) {
            return optional($this->jumpCloud)->platform;
        }

        return '';
    }

    public function getEndPointImeiAttribute()
    {
        if ($this->airwatch) {
            return optional($this->airwatch)->imei;
        }
        if ($this->mobileIron) {
            return optional($this->mobileIron)->imei;
        }

        return '';
    }

    public function getEndPointOsAttribute()
    {
        if ($this->intune) {
            return optional($this->intune)->os;
        }
        if ($this->jamf) {
            return optional($this->jamf)->os_build;
        }
        if ($this->kandjiDevices) {
            return optional($this->kandjiDevices)->os;
        }

        if ($this->jumpCloud) {
            return optional($this->jumpCloud)->os;
        }

        return '';
    }

    public function getEndPointOsVersionAttribute()
    {
        if ($this->intune) {
            return optional($this->intune)->os_version;
        }
        if ($this->jamf) {
            return optional($this->jamf)->os_version;
        }
        if ($this->chromebook) {
            return optional($this->chromebook)->os_version;
        }
        if ($this->kandjiDevices) {
            return optional($this->kandjiDevices)->os_version;
        }

        if ($this->jumpCloud) {
            return optional($this->jumpCloud)->os_version;
        }

        return '';
    }

    public function getEndPointProcessorSpeedAttribute()
    {
        if ($this->jamf) {
            return optional($this->jamf)->processor_speed;
        }
        if ($this->chromebook) {
            return convertSpecsToReadableValue(optional($this->chromebook)->processor_speed, 'processor_speed');
        }
        if ($this->kandjiDevices) {
            return optional($this->kandjiDevices)->processor_speed;
        }

        if ($this->jumpCloud) {
            return optional($this->jumpCloud)->processor_speed;
        }

        return '';
    }

    public function getEndPointMemoryAttribute()
    {
        if ($this->jamf) {
            return optional($this->jamf)->ram;
        }
        if ($this->chromebook) {
            return convertSpecsToReadableValue(optional($this->chromebook)->ram, 'ram');
        }
        if ($this->kandjiDevices) {
            return optional($this->kandjiDevices)->ram;
        }

        if ($this->jumpCloud) {
            return optional($this->jumpCloud)->ram;
        }

        return '';
    }
    public function getEndPointDiskCapacityAttribute()
    {
        if ($this->intune) {
            return convertSpecsToReadableValue(optional($this->intune)->disk_capacity, 'disk_capacity');
        }
        if ($this->jamf) {
            return optional($this->jamf)->disk_capacity;
        }
        if ($this->airwatch) {
            return '';
        }
        if ($this->chromebook) {
            return convertSpecsToReadableValue(optional($this->chromebook)->disk_capacity, 'disk_capacity');
        }
        if ($this->kandjiDevices) {
            return optional($this->kandjiDevices)->disk_capacity;
        }
        if ($this->mobileIron) {
            return optional($this->mobileIron)->totalCapacity;
        }
        if ($this->jumpCloud) {
            return optional($this->jumpCloud)->disk_capacity;
        }

        return '';
    }

    public function getEndPointEncryptedAttribute()
    {
        if ($this->intune) {
            return optional($this->intune)->encrypted;
        }

        return '';
    }

    public function getEndPointActivationLockAttribute()
    {
        if ($this->intune) {
            return optional($this->intune)->activation_lock;
        }

        if ($this->kandjiDevices) {
            return optional($this->kandjiDevices)->activation_lock;
        }

        return '';
    }

    /**
     * get Source
     */
    public function getEndPointSourceAttribute()
    {
        if ($this->intune) {
            return "Intune";
        }
        if ($this->jamf) {
            return "Jamf";
        }
        if ($this->airwatch) {
            return "Workspace ONE";
        }
        if ($this->chromebook) {
            return "Chromebook";
        }
        if ($this->kandjiDevices) {
            return "Kandji";
        }
        if ($this->mobileIron) {
            return "MobileIron";
        }

        if ($this->jumpCloud) {
            return "JumpCloud";
        }
    }

    public function cdwAccessoriesMapping()
    {
        return $this->hasOne('App\Models\AsnAccessoriesMapping', 'description', 'description');
    }


    public function scopeIsComputer($query)
    {
        return $query->whereHas('assetType', function ($query) {
            $query->whereIn('slug', ['desktop', 'computer', 'laptop']);
        });
    }

    public function scopeDoesNotHaveAssetStatus($query, array $slugs)
    {
        return $query->whereHas('assetStatus', function ($query) use ($slugs) {
            $query->whereNotIn('slug', $slugs);
        });
    }

    public function getWarrantyEndDateAttribute()
    {
        return parse_date_from_db_date($this->attributes['warranty_end_date']);
    }

    //find all assets details of user with specified status
    public function scopeTermAsset($query)
    {
        return $query->whereIn(
            'asset_status_id',
            AssetStatus::whereIn('slug', ['assigned', 'loaned', 'pending_return'])->pluck('id')
        )
            ->whereHas('assetType', function ($query) {
                $query->whereNotIn('slug', ['mobile_phone_number']);
            });
    }

    /**
     * Assets which are not having any discovery tools.
     * @param mixed $query
     *
     * @return [type]
     */
    public function scopeHasNoDiscoveryAssets($query)
    {
        return $query->doesntHave('airwatch')
            ->doesntHave('chromebook')
            ->doesntHave('intune')
            ->doesntHave('jamf')
            ->doesntHave('KandjiDevices')
            ->doesntHave('mobileIron')
            ->doesntHave('jumpCloud');
    }

    /*
      Functions to get history details
    */
    public function assetHistoryNewestFirst()
    {
        return $this->hasMany('App\Models\AssetHistory')->orderBy('created_at', 'desc');
    }

    public function assetHistoryStatusUpdated()
    {
        return $this->hasOne('App\Models\AssetHistory')->where('action', 'status_updated')->latest();
    }

    public function assetHistoryCreated()
    {
        return $this->hasOne('App\Models\AssetHistory')->where('action', 'created')->latest();
    }

    /**
     * Format last_status_upadate
     * @param mixed $date
     *
     * @return date
     */
    public function getLastStatusUpdateAttribute($date)
    {
        return $date ? parse_date_from_db_datetime($date) : '';
    }

    /**
     * Exclude assets if thier asset status id in given list of ids.
     *
     * @param mixed $query
     * @param mixed $assetStatuses
     *
     * @return mixed $query
     */
    public function ScopeExcludeAssetsByStatus($query, $assetStatuses)
    {
        if (is_array($assetStatuses)) {
            return $query->whereNotIn('asset_status_id', $assetStatuses);
        }

        return $query->where('asset_status_id', '!=', $assetStatuses);
    }

    /**
     * Exclude assets if thier asset type id in given list of ids.
     *
     * @param mixed $query
     * @param mixed $assetTypes
     *
     * @return mixed $query
     */
    public function ScopeExcludeAssetsByAssetType($query, $assetTypes)
    {
        if (is_array($assetTypes)) {
            return $query->whereNotIn('asset_type_id', $assetTypes);
        }

        return $query->where('asset_type_id', '!=', $assetTypes);
    }

    /**
     * Get assets by their asset tag or serial #.
     *
     * @param mixed $query
     * @param mixed $assetIdentifiers
     *
     * @return $query
     */
    public function scopeGetAssetByAssetTagsOrSerialNumbers($query, $assetIdentifiers)
    {
        return $query->where(function ($query) use ($assetIdentifiers) {
            $query->whereIn(DB::raw('lower(asset_tag)'), array_map('strtolower', $assetIdentifiers))
                ->orWhereIn(DB::raw('lower(serial_no)'), array_map('strtolower', $assetIdentifiers));
        });
    }

    /**
     * @param mixed $query
     * @param int   $days
     *                     Remove assets  with status stolen_lost  for more than '$days' days
     *
     * @return [type]
     */
    public function scopeFilterOutStolenOrLostAssetsDatedMoreThanSpecifiedDays($query, $days = 30)
    {
        return $query->whereNotIn('id', function ($query) use ($days) {
            $query->select('id')->from('assets')->where('asset_status_id', '=', AssetStatus::where('slug', 'stolen_lost')->pluck('id'))
                ->whereDate('lost_date', '<=', Carbon::now()->subDays($days)->toDateString());
        });
    }

    /**
     * Exclude assets by their asset tag or serial #.
     *
     * @param mixed $query
     * @param mixed $assetIdentifiers
     *
     * @return $query
     */
    public function scopeExcludeAssetsByAssetTagsOrSerialNumbers($query, $assetIdentifiers)
    {
        return $query->where(function ($query) use ($assetIdentifiers) {
            $query->whereNotIn(DB::raw('lower(asset_tag)'), array_map('strtolower', $assetIdentifiers))
                ->whereNotIn(DB::raw('lower(serial_no)'), array_map('strtolower', $assetIdentifiers));
        });
    }

    /**
     * Get assets by their location.
     *
     * @param mixed $query
     * @param mixed $locationId
     *
     * @return $query
     */
    public function scopeGetAssetsByLocationId($query, $locationId)
    {
        return $query->where('location_id', $locationId);
    }

    public function scopeAssignedAndLoaned($query)
    {
        return $query->whereHas('assetStatus', function ($query) {
            $query->where('slug', 'assigned')
                ->orWhere('slug', 'loaned');
        });
    }

    public function scopeWithoutTechnicalSpec($query)
    {
        return $query->where(function ($query) {
            return $query->whereNull('technical_spec_id')
                ->orWhere('technical_spec_id', 0);
        });
    }

    public function vendor()
    {
        return $this->belongsTo('App\Models\Vendor');
    }

    public function assetVulnerabilty()
    {
        return $this->hasMany('App\Models\AssetVulnerability');
    }

    /**
     * Retrieves the value of the "sensor_detected" attribute.
     *
     * @return string Returns "Yes" if the "crowdstrike_id" attribute is not empty, otherwise returns "No".
     */
    public function getSensorDetectedAttribute()
    {
        return !empty($this->getAttribute('crowdstrike_id'))  ? 'Yes' : 'No';
    }

    public function purchaseOrder()
    {
        return $this->belongsTo('App\Models\PurchaseOrder');
    }

    public function workspaceOne()
    {
        return $this->hasOne('App\Models\AirWatch');
    }

    /**
     * Retrieves the latest assigned asset history.
     *
     * @return \App\Models\AssetHistory The latest assigned asset history.
     */
    public function latestAssignedAssetHistory()
    {
        return $this->hasOne(\App\Models\AssetHistory::class)->where('action', 'status_updated')->where('new_value', 'Assigned')->latest();
    }

    /**
     * Retrieves the latest lost/stolen asset history.
     *
     * @return \App\Models\AssetHistory
     */
    public function latestLostOrStolenAssetHistory()
    {
        return $this->hasOne(\App\Models\AssetHistory::class)->where('new_asset_status_id', AssetStatus::where('slug', 'stolen_lost')->pluck('id'))->latest();
    }

    /**
     * Get the history of reused assets.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function reusedAssetHistory(): HasMany
    {
        $stockStatuses = AssetStatus::whereIn('slug', config('reports.reused_devices.stock_statuses'))->pluck('id')->toArray();
        $usedStatuses = AssetStatus::whereIn('slug', config('reports.reused_devices.used_statuses'))->pluck('id')->toArray();

        return $this->hasMany('App\Models\AssetHistory')->where('action', 'status_updated')->whereNotIn('old_asset_status_id', $stockStatuses)->whereIn('new_asset_status_id', $usedStatuses);
    }

    public function scopeGetAssetAssignedDate($query)
    {
        return $query->addSelect([
            'assigned' => AssetHistory::select('created_at')->whereColumn('asset_id', 'assets.id')->whereNotNull('new_user_id')->orderBy('created_at', 'asc')->limit(1),
        ]);
    }

    public function scopeIsLaptopAndComputer($query)
    {
        return $query->whereHas('assetType', function ($query) {
            $query->whereIn('slug', ['laptop', 'computer']);
        });
    }

    /**
     * Retrieves the latest asset history for the assigned or installed assets.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function latestAssignedOrInstalledAssetHistory()
    {
        return $this->hasOne('App\Models\AssetHistory')->whereIn('action', ['status_updated', 'created', 'asset_received'])->whereIn('new_value', ['Assigned', 'Installed'])->latest();
    }

    /**
     * Retrieves the latest asset history of the assets based on their status.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function latestAssetHistoryBasedOnStatus($status)
    {
        $relation = $this->hasOne('App\Models\AssetHistory')->whereIn('action', ['status_updated', 'created', 'asset_received', 'linked', 'moved', 'returned']);
        if (is_array($status)) {
            $relation->whereIn('new_asset_status_id', $status);
        } else {
            $relation->where('new_asset_status_id', $status);
        }
        return $relation->latest()->first();
    }

    /**
     * Scope to get assets received from terminated users.
     *
     * @param mixed $query
     */
    public function scopeReceivedAsset($query)
    {
        return $query->leftJoin('users', 'users.id', '=', 'assets.user_id')
            ->where(function ($query) {
                $query->whereIn('asset_status_id', AssetStatus::whereNotIn('slug', ['assigned', 'loan_or_test', 'uncollected_term'])->pluck('id'))
                    ->orWhere(function ($query) {
                        $query->where('asset_status_id', AssetStatus::where('slug', 'stolen_lost')->pluck('id'))
                            ->whereHas('latestAssetHistory', function ($query) {
                                $query->whereRaw('updated_at + INTERVAL 10 DAY < users.terminated_date');
                            });
                    });
            })->whereHas('assetType', function ($query) {
                $query->whereNotIn('slug', ['mobile_phone_number']);
            });
    }

    /**
     * Retrieves the original value of the asset.
     *
     * if the asset original value is null, returns the original value of the technical spec.
     */
    public function getOriginalValueAttribute()
    {
        if ($this->asset_original_value) {
            return $this->asset_original_value;
        }

        return $this->technicalSpec ? $this->technicalSpec->original_value : 0;
    }

    /**
     * Retrieve the RetrieverDevice associated with this Asset.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function retrieverDevice()
    {
        return $this->hasOne('App\Models\RetrieverDevice');
    }

    /**
     * Relation to AssetHistory table to get status update history by Retriver API
     */
    public function retrieverUpdateHistory()
    {
        return $this->hasOne('App\Models\AssetHistory')->retrieverAssetUpdate();
    }

    /* Get the associated RevivnAsset.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function revivnAsset()
    {
        return $this->hasOne('App\Models\RevivnAsset');
    }

    /**
     * Scope a query to only include assets due yesterday.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query The query builder instance.
     * @throws \Some_Exception_Class Description of the exception.
     * @return \Illuminate\Database\Eloquent\Builder The modified query builder instance.
     */
    public function scopeLoanerOverDueAssets($query)
    {
        return $query->hasStatus('loaned')
            ->whereDate('loaner_return_date', '<=', Carbon::yesterday()->toDateString());
    }

    /**
     * Relation to JumpCloudDevices table to get JumpCloud device data.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function jumpCloud()
    {
        return $this->hasOne('App\Models\JumpCloudDevices');
    }

    /**
     * Scope a query to only include assets due next week.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query The query builder instance.
     * @return \Illuminate\Database\Eloquent\Builder The modified query builder instance.
     */
    public function scopeLoanerDueNextWeekAssets($query)
    {
        return $query->hasStatus('loaned')
            ->whereDate('loaner_return_date', Carbon::now()->addDay(7)->toDateString());
    }

    /**
     * Retrieve the associated Zoom model.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne The has one relationship.
     */
    public function zoom()
    {
        return $this->hasOne('App\Models\ZoomDevice');
    }
}
