<?php

namespace App\Listeners;

use Aacotroneo\Saml2\Events\Saml2LoginEvent;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use App\User;

class Saml2LoginListener
{
    /**
     * Create the event listener.
     *
     * @return void
     */
    public function __construct()
    {
        // Initialization if needed
    }

    /**
     * Handle the SAML2 login event.
     *
     * @param  Saml2LoginEvent  $event
     * @return void
     */
    public function handle(Saml2LoginEvent $event)
    {
        $user = $event->getSaml2User();

        $attributes = $user->getAttributes();
        $email = $this->extractUserEmail($attributes);

        if (empty($email)) {
            $errorMessage = 'No valid email address found';
            Log::error($errorMessage, ['attributes' => array_keys($attributes)]);
            session(['saml2_error_single' => 'Your account email address could not be found. Please contact your administrator.']);
            return;
        }

        $laravelUser = User::where('email', $email)->first();

        if (!$laravelUser) {
            $errorMessage = 'User not found';
            Log::warning($errorMessage, ['email' => $email]);
            session(['saml2_error_single' => 'You do not have access to this system. Please contact support.']);
            return;
        }

        try {
            Auth::login($laravelUser);
        } catch (\Exception $e) {
            Log::error('Failed to authenticate user via SAML', [
                'error' => $e->getMessage(),
                'email' => $email,
                'trace' => $e->getTraceAsString()
            ]);
            session(['saml2_error_single' => 'We encountered an issue logging you in. Please try again or contact support.']);
        }
    }

    /**
     * Extract user email from SAML attributes
     *
     * @param array $attributes
     * @return string|null
     */
    protected function extractUserEmail(array $attributes): ?string
    {
        $possibleEmailKeys = [
            'http://schemas.xmlsoap.org/ws/2005/05/identity/claims/name',
            'http://schemas.xmlsoap.org/ws/2005/05/identity/claims/emailaddress',
            'email',
            'mail',
            'User.email',
            'User.mail'
        ];

        foreach ($possibleEmailKeys as $key) {
            if (!empty($attributes[$key][0])) {
                return filter_var($attributes[$key][0], FILTER_SANITIZE_EMAIL);
            }
        }

        return null;
    }
}