<?php

namespace App\Http\Traits;

use App\Models\SavedSearch;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;

trait SavedSearchTrait
{
    /**
     * Save the searches
     * @return [type]
     */
    public function saveSearch()
    {
        try {
            $parameters = [];
            $name = Carbon::now();
            if (request('name')) {
                $name = request('name');
            }


            request()->validate([
                'name' => [Rule::unique('saved_searches')->where(function ($query) {
                    return $query->where(['user_id' => auth()->id(), 'view_all' => request('view_all')]);
                })],
            ]);

            $parameters = $this->getSavedSearchParameters();
            $savedSearch = SavedSearch::create([
                'report_name' => $this->getSavedSearchReportName(),
                'user_id' => Auth::id(),
                'name' => $name,
                'view_all' => request('view_all'),
                'parameters' => json_encode($parameters),
            ]);

            return response()->json(['status' => 'success', 'message' => 'Search saved successfully.', 'dropdown' => '<option value=' . $savedSearch->id . '>' . $savedSearch->name . '</option>']);
        } catch (ValidationException) {
            return response()->json(['status' => 'error', 'message' => 'The search name cannot be duplicated.']);
        } catch (Exception $e) {
            return response()->json(['status' => 'error', 'message' => 'Some error occurred.']);
        }
    }


    /**
     * Updates a saved search
     * @return [type]
     */
    public function updateSearch()
    {
        $parameters = [];
        $reportName = Carbon::now();
        if (request('name')) {
            $reportName = request('name');
        }


        $parameters = $this->getSavedSearchParameters();
        try {

            request()->validate([
                'name' => [Rule::unique('saved_searches')->where(function ($query) {
                    return $query->where(['user_id' => auth()->id(), 'view_all' => request('view_all')]);
                })->ignore(request('report_id'))],
            ]);

            $report = SavedSearch::findOrFail(request('report_id'));
            $oldViewAll = $report->view_all;
            if ($report->user_id != Auth::id()) {
                request()->session()->flash('error', 'You don’t have permission to update this search');
                return response()->json('permission error');
            }
            $report->update([
                'name' => $reportName,
                'view_all' => request('view_all'),
                'parameters' => json_encode($parameters),
            ]);
            $url = '';
            if ($oldViewAll != $report->view_all) {
                $url = $report->view_all ? ('?admin_saved_search=' . $report->id) : ('?user_saved_search=' . $report->id);
            }
            request()->session()->flash('message', 'Search updated successfully.');
            return response()->json(['status' => 'success', 'url' => $url]);
        } catch (ValidationException) {
            return response()->json(['status' => 'error', 'message' => 'The search name cannot be duplicated.']);
        } catch (Exception $e) {
            request()->session()->flash('error', 'Search not found.');
            return response()->json('error');
        }
    }

    /**
     * Delete a search
     * @return [type]
     */
    public function deleteSearch()
    {
        try {
            $report = SavedSearch::findOrFail(request('report_id'));
            if ($report->user_id != Auth::id()) {
                request()->session()->flash('error', 'You don’t have permission to delete this search.');
                return response()->json('permission error');
            }

            $report->delete();
            request()->session()->flash('message', 'Search deleted successfully.');
            return response()->json('success');
        } catch (Exception $e) {
            request()->session()->flash('error', 'Search not found.');
            return response()->json('error');
        }
    }
}
