<?php

namespace App\Http\Traits;

use Illuminate\Support\Str;

trait GeneratesSecureToken
{
    /**
     * Generate a secure token.
     *
     * @param  int  $length
     * @return string
     */
    public function generateSecureToken($length = 32)
    {
        $token = Str::random($length);

        // Add additional security features
        $token = $this->encryptToken($token);
        $token = $this->addEntropy($token);
        $hashedToken = $this->hashToken($token);

        $tokenWithoutSymbols = $this->removeSymbols($hashedToken);

        return substr($tokenWithoutSymbols, 0, $length);
    }

    /**
     * Encrypt the token for added security.
     *
     * @param  string  $token
     * @return string
     */
    protected function encryptToken($token)
    {
        return encrypt($token);
    }

    /**
     * Add entropy to the token for increased randomness.
     *
     * @param  string  $token
     * @return string
     */
    protected function addEntropy($token)
    {
        $entropy = Str::random(10);

        return $token . $entropy;
    }

    /**
     * Hash the token for one-way encryption.
     *
     * @param  string  $token
     * @return string
     */
    protected function hashToken($token)
    {
        return hash('sha256', $token);
    }

    /**
     * Remove symbols from the token.
     *
     * @param  string  $token
     * @return string
     */
    protected function removeSymbols($token)
    {
        return preg_replace('/[^a-zA-Z0-9]/', '', $token);
    }
}
