<?php

namespace App\Http\Traits;

use App\Models\Asset;
use Carbon\Carbon;
use Facades\App\Repositories\BulkUpload;

trait BulkUploadTrait
{
    /**
     * Convert CSV to array.
     *
     * @param string $path
     *
     * @return mixed
     */
    public function getItems($path)
    {
        if (!file_exists($path)) {
            return false;
        }

        $extension = getFileExtension($path);

        if ($extension == 'csv') {
            $items = csv_to_array($path);
        } elseif ($extension == 'xlsx') {
            $items = excel_to_array($path);
        }

        if (empty($items)) {
            return false;
        }

        return $items;
    }

    public function getExcelItem(string $path)
    {
        if (!file_exists($path)) {
            return false;
        }

        $items = excel_to_array($path);

        if (empty($items)) {
            return false;
        }

        return $items;
    }

    /**
     * Retrieves relational values for specified fields from the item.
     *
     * @param array  $item       The item data from the CSV file.
     * @param string $dbField    The database field to be processed.
     * @param string $fileHeader The corresponding file header from the CSV.
     *
     * @return mixed The relational value for the field, or false if no value is found.
     */
    public function getRelationalValues($item, $dbField, $fileHeader)
    {
        $methods = [
            'location_id' => [BulkUpload::class, 'getLocationId'],
            'ticket_no' => fn() => $item[$fileHeader] ?? null,
            'user_id' => [BulkUpload::class, 'getUserId'],
            'asset_status_id' => [BulkUpload::class, 'getStatusId'],
            'make_and_model_id' => [BulkUpload::class, 'getAttributeMakeModelId'],
            'technical_spec_id' => [BulkUpload::class, 'getAttributeTechnicalSpecId'],
            'asset_type_id' => [BulkUpload::class, 'getAssetTypeId'],
            'carrier_id' => [BulkUpload::class, 'getCarrierId'],
            'asset_tracking_id' => [BulkUpload::class, 'getAssetTrackingId'],
            'vendor_id' => [BulkUpload::class, 'getVendorId'],
        ];

        if (!isset($methods[$dbField])) {
            return false;
        }

        $method = $methods[$dbField];

        return is_callable($method)
            ? $method($item, $dbField, $fileHeader)
            : call_user_func($method, $item, $dbField, $fileHeader);
    }


    /**
     * Checks if there are any duplicate serial numbers in the upload data.
     *
     * @param array $csvData The upload data.
     *
     * @return array An array containing the errors and the valid data.
     */
    public function checkRepeatUploadData($csvDatas)
    {
        $errors = [];
        $validData = [];

        $csvDatas = collect($csvDatas);

        $duplicates = $csvDatas->groupBy(function ($item) {
            return $item['assetData']['serial_no'];
        })->filter(function ($group) {
            return $group->count() > 1;
        });

        foreach ($duplicates as $serialNo => $group) {
            foreach ($group as $data) {
                $errors[] = 'Line no ' . ($data['assetData']['asset_count'] ?? 0 + 1) . ' : The serial # ' . $serialNo . ' has duplicates.';
            }
        }

        $validData = $csvDatas->filter(function ($data) use ($duplicates) {
            return !$duplicates->has($data['assetData']['serial_no']);
        })->values()->toArray();

        return ['errors' => $errors, 'csvData' => $validData];
    }

    /**
     * Validate Each CSV fields
     * @param  array $csvData
     * @return array
     */
    public function validateFields(array $csvData, $type = null)
    {
        $count = 2;
        $errors = [];
        $validData = [];
        foreach ($csvData as $data) {
            $error = $this->csvValidator($data['assetData'], $count++, $type);
            if ($error !== null) {
                $errors[] = $error;
                continue;
            }
            $validData[] = $data;
        }

        return ['errors' => $errors, 'csvData' => $validData]; //$errors;
    }

    /**
     * Handles import errors by checking if the provided data contains any errors.
     *
     * @param array $data The data to check for errors.
     * @return array|null An array containing the rendered view for errors, or null if there are no errors.
     */
    public function handleImportErrors($data)
    {
        if (!empty($data['error'])) {
            $errors = $data['error'];
            return ['errors' => view('assets.partials.upload-csv-errors', compact('errors'))->render()];
        }

        return null;
    }

    /**
     * Handles CSV column errors based on the provided path and CSV data.
     *
     * @param  $path Description of the path parameter.
     * @param  $csvData Description of the csvData parameter.
     * @param  $headerKeys Description of the headerKeys parameter.
     * 
     * @return Some_Return_Value Description of the return value.
     */
    public function handleCsvColumnErrors($path, $csvData, $headerKeys = null)
    {
        if (empty($headerKeys)) {
            $headerKeys = array_values($this->getHeaderMap());
        }

        $errors = $this->validateCsvColumns($path, $csvData, $headerKeys);
        if ($errors != '') {
            return ['errors' => view('assets.partials.upload-csv-errors', compact('errors'))->render()];
        }

        return null;
    }

    /**
     * Validate Columns have data
     * @param  string $path
     * @param  array  $csvData
     * @param  array  $headerKeys
     * @return array
     */
    public function validateCsvColumns(string $path, array $csvData, $headerKeys)
    {
        $extension = getFileExtension($path);

        if ($extension == 'csv') {
            $csv = array_map("str_getcsv", file($path, FILE_SKIP_EMPTY_LINES));
            $csvKeys = array_shift($csv);
        } elseif ($extension == 'xlsx') {
            $csvKeys = get_excel_header($path);
        }

        if (empty($csvData)) {
            return 'Please make sure the file contains data';
        }

        $messages = $this->checkFileDifference($csvKeys, $headerKeys);

        return $messages;
    }

    /**
     * Check csv file difference from the template file
     * @param  array $csvKeys
     * @param  array $headerKeys
     * @return string
     */
    public function checkFileDifference(array $csvKeys, array $headerKeys)
    {
        if (count($headerKeys) >= count($csvKeys)) {
            $result = array_diff($headerKeys, $csvKeys);
            $messageString = 'Please make sure the column names are spelled correctly. Columns missing - ';
        } else {
            $result = array_diff($csvKeys, $headerKeys);
            $messageString = 'Uploaded file is not match with template file. Extra columns - ';
        }
        $messages = '';

        if (!empty($result)) {
            foreach ($result as  $value) {
                $messageString .= $value . ', ';
            }
            $messages = rtrim($messageString, ", ");
        }

        return $messages;
    }

    /**
     * Get Additional History Data.
     *
     * @param array $assetHistory [description]
     * @param array $assetData    [description]
     * @param Asset $asset        [description]
     *
     * @return [type] [description]
     */
    public function getAdditionalHistoryData(array $assetHistory, array $assetData, Asset $asset)
    {
        $keyArray = ['loaner_return_date', 'location_id', 'user_id', 'loaner_retention_date', 'asset_status_id'];

        foreach ($keyArray as $key) {
            if (array_key_exists($key, $assetData)) {
                $assetHistory['new_' . $key] = $assetData[$key];
                $assetHistory['old_' . $key] = $asset->$key;
            }
        }

        if (array_key_exists('wipe_confirmation', $assetData)) {
            $assetHistory['wipe_confirmation'] = $assetData['wipe_confirmation'];
        }

        return $assetHistory;
    }

    /**
     * Get Additional Data based on asset status.
     *
     * @return array
     */
    public function getAdditionalStatusData(array $data)
    {
        if ($data['status_slug'] == 'loaned') {
            $data['loaner_return_date'] = isset($data['loaner_return_date']) ? $data['loaner_return_date'] : parse_date_from_db_datetime(Carbon::now()->addDays(config('date.period.loaner')));
        }

        if (in_array($data['status_slug'], ['lost_or_stolen', 'stolen_lost'])) {
            $data['lost_date'] = isset($data['lost_date']) ? $data['lost_date'] : date('m/d/Y');
        }

        if ($data['status_slug'] == 'retain_hold') {
            $data['loaner_retention_date'] = isset($data['loaner_retention_date']) ? $data['loaner_retention_date'] : parse_date_from_db_datetime(Carbon::now()->addDays(config('date.period.retention', '')));
        }

        return $data;
    }
}
