<?php

namespace App\Http\Traits;

use App\Events\BulkUpdates;
use App\Models\Asset;
use App\Models\AssetStatus;
use App\Models\AssetType;
use App\Models\Location;
use App\Models\MakeAndModel;
use App\Models\Manufacturer;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Facades\App\Services\AssetHistory as AssetHistoryService;

trait AutoCreateInstallTrait
{
    /**
     * Creates and installs an asset based on discovery data.
     *
     * @param mixed $discoveryData Data from the discovery tool for the asset.
     * @param string $discoveryTool The name of the discovery tool used for this operation.
     *
     * @return void
     */
    public function autoCreateInstall($discoveryData, $discoveryTool)
    {
        $locationId = $discoveryData['location_id'];
        $assetData = $this->getAssetData($discoveryData, $locationId);

        if (!$assetData) {
            Log::channel('daily')->error('Asset data not found');

            return;
        }
        $assetId = Asset::firstOrCreate($assetData)->id;
        $location = Location::find($locationId);

        $description = __('history.AssetCreatedAndInstalled', [
            'assetName'   => $assetData['serial_no'],
            'assetId'     => $assetId,
            'roomName'    => $location->room_name,
        ]);

        $installedAssetStatus = Cache::remember('installed_asset_status', 3600, function () {
            return AssetStatus::getStatusID('installed')->first();
        });

        $assetHistory = [
            'asset_id'            => $assetId,
            'action'              => 'created',
            'new_asset_status_id' => optional($installedAssetStatus)->id,
            'new_value'           => optional($installedAssetStatus)->name,
            'new_location_id'     => $locationId,
            'description'         => $description,
            'created_by'          => $discoveryTool,
            'created_at'          => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'          => Carbon::now()->format('Y-m-d H:i:s'),
        ];

        event(new BulkUpdates($assetHistory));

        $this->updateDiscoveryAssetId($discoveryData, $assetId);
    }

    /**
     * Constructs an array of asset data for creating an asset
     *
     * @param \stdClass $discoveryData Zoom device data
     * @param int $locationId ID of the location where the asset is installed
     *
     * @return array Constructed asset data
     */
    public function getAssetData($discoveryData, $locationId)
    {
        if (empty($discoveryData['serial_no'])) {
            return false;
        }

        $assetTypeId    = $this->getAssetTypeId($discoveryData['device_type']);
        $manufacturerId = $this->getManufacturerId($discoveryData['manufacturer']);
        $makeModelId    = $this->getMakeModelId($manufacturerId, $assetTypeId, $discoveryData['model']);

        if (!$makeModelId) {
            return false;
        }

        $installedAssetStatus = Cache::remember('installed_asset_status', 3600, function () {
            return AssetStatus::getStatusID('installed')->first();
        });

        $assetData = [
            'serial_no'         => $discoveryData['serial_no'],
            'asset_status_id'   => $installedAssetStatus->id,
            'asset_type_id'     => $assetTypeId,
            'make_and_model_id' => $makeModelId,
            'location_id'       => $locationId,
        ];

        return $assetData;
    }

    /**
     * Retrieves the ID of the given asset type, creating it if it doesn't exist yet.
     *
     * @param  string  $assetType
     *
     * @return int
     */
    public function getAssetTypeId($assetType)
    {
        if (empty($assetType)) {
            return null;
        }

        return AssetType::firstOrCreate(['name' => $assetType], ['slug' => str_slug($assetType, '_')])->id;
    }

    /**
     * Retrieves the ID of the given manufacturer, creating it if it doesn't exist yet.
     *
     * @param  string  $manufacturer
     *
     * @return int
     */
    public function getManufacturerId($manufacturer)
    {
        if (empty($manufacturer)) {
            return null;
        }

        return Manufacturer::firstOrCreate(['name' => $manufacturer], ['slug' => str_slug($manufacturer, '_')])->id;
    }

    /**
     * Retrieves the ID of the given make and model, creating it if it doesn't exist yet.
     *
     * @param  int    $manufacturerId
     * @param  int    $assetTypeId
     * @param  string $makeModel
     *
     * @return int
     */
    public function getMakeModelId($manufacturerId, $assetTypeId, $makeModel)
    {
        if (empty($makeModel) || !$manufacturerId || !$assetTypeId) {
            return null;
        }

        return MakeAndModel::firstOrCreate([
            'name' => $makeModel,
            'manufacturer_id' => $manufacturerId,
            'asset_type_id' => $assetTypeId
        ], [
            'slug' => str_slug($makeModel, '_'),
        ])->id;
    }

    /**
     * Re-installs the assets from the location discrepancy report.
     *
     * @param mixed $discoveryAssets Array of discovery data
     * @param string $discoveryTool Name of the discovery tool
     *
     * @return bool
     */
    public function reInstallDiscrepancyAssets($discoveryAssets, $discoveryTool)
    {
        try {
            $installedAssetStatus = Cache::remember('installed_asset_status', 3600, function () {
                return AssetStatus::getStatusID('installed')->first();
            });

            foreach ($discoveryAssets as $discoveryAsset) {
                $asset = Asset::find($discoveryAsset->asset_id);

                if (!$asset) {
                    continue;
                }

                $this->createAutoInstalledAssetHistory($discoveryAsset, $discoveryTool);
                $asset->update([
                    'location_id' => $discoveryAsset->location_id,
                    'asset_status_id' => $installedAssetStatus->id,
                ]);
            }
        } catch (Exception $e) {
            Log::channel('daily')->error($e->getMessage());
        }

        return true;
    }

    /**
     * Creates an asset history entry for an auto installed asset.
     *
     * @param \stdClass $discoveryAsset Asset data from the discovery tool
     * @param string $discoveryTool Name of the discovery tool
     *
     * @return void
     */
    public function createAutoInstalledAssetHistory($discoveryAsset, $discoveryTool)
    {
        $installedAssetStatus = Cache::remember('installed_asset_status', 3600, function () {
            return AssetStatus::getStatusID('installed')->first();
        });
        $description = AssetHistoryService::getAutoInstalledAssetHistoryDescription($discoveryAsset);
        $assetHistory = [
            'asset_id'            => $discoveryAsset->asset_id,
            'action'              => 'auto_installed',
            'old_location_id'     => $discoveryAsset->asset?->location_id,
            'new_location_id'     => $discoveryAsset->location_id,
            'old_asset_status_id' => $discoveryAsset->asset?->asset_status_id,
            'new_asset_status_id' => $installedAssetStatus->id,
            'old_user_id'         => $discoveryAsset->asset?->user_id,
            'description'         => $description,
            'created_by'          => $discoveryTool,
            'created_at'          => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'          => Carbon::now()->format('Y-m-d H:i:s'),
        ];

        event(new BulkUpdates($assetHistory));
    }
}
