<?php

namespace App\Http\Traits;

use App\Events\BulkUpdates;
use App\Models\Asset;
use App\Models\AssetStatus;
use App\Models\Carrier;
use App\Models\DiscoveryToolsHardwareMapping;
use App\Models\MakeAndModel;
use App\Models\Manufacturer;
use App\Models\TechnicalSpecs;
use App\User;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Facades\App\Services\AssetHistory as AssetHistoryService;

trait AutoCreateAssignTrait
{

    /**
     * Get the Tech spec for creating asset
     */
    public function getTechSpec($discoveryData, $field = null)
    {
        $makeAndModelMapping = DiscoveryToolsHardwareMapping::select('make_and_model_id')
            ->with('makeAndModel:id,asset_type_id')
            ->where('mdm_name', $field)
            ->where('mdm_model', $discoveryData['model'])->first();

        if (!$makeAndModelMapping) {
            $makeModelId = $this->createHardwareStandard($discoveryData);
            $makeAndModelMapping = '';
        } else {
            $makeModelId = $makeAndModelMapping->make_and_model_id;
        }

        $techSpec = $this->getTechSpecString($discoveryData);
        $technicalSpecs = TechnicalSpecs::firstOrCreate([
            'details'  => $techSpec,
            'make_and_model_id' => $makeModelId
        ], [
            'original_value' => 0,
            'original_value' => 0,
            'auto_created' => 1,
        ]);

        return array('makeAndModelMapping' => $makeAndModelMapping, 'technicalSpecs' => $technicalSpecs);
    }

    /**
     * Create manufacure, makeandmodel if hardwarestandard exists
     */
    public function createHardwareStandard($data)
    {
        $hardwareStandardId = $this->checkHardwareStandardExist($data['model']);

        if ($hardwareStandardId) {
            return  $hardwareStandardId;
        }

        $manufacturerId =  Manufacturer::updateOrCreate(['slug' => str_slug($data['manufacturer'], '_'), 'name' => $data['manufacturer']])->id;

        $hardwareData['manufacturer_id'] = $manufacturerId;
        $hardwareData['asset_type_id']   = $data['asset_type_id'];
        $hardwareData['name'] = $data['model'];
        $hardwareData['slug'] = str_slug($data['model'], '_');

        return MakeAndModel::updateOrCreate($hardwareData)->id;
    }

    /**
     * Check if the hw standard exist
     * @param mixed $makeModel
     *
     * @return [type]
     */
    public function checkHardwareStandardExist($makeModel)
    {
        $exist = MakeAndModel::select('id')->where('name', $makeModel)->first();

        if (!$exist) {
            return false;
        }

        return $exist->id;
    }

    /**
     * Create and assign assets to  users
     */
    public function autoCreateAndAssign($discoveryData, $techSpec, $discoveryTool = null)
    {
        $user = User::select('id', 'first_name', 'last_name', 'email', 'status')->where('id', $discoveryData['user_id'])->first();

        // If user is not active skip assigning
        if ($user->status !== 1) {
            return false;
        }

        $assetData = $this->getAssetData($discoveryData, $techSpec['technicalSpecs']);
        $assetId = Asset::firstOrCreate($assetData)->id;

        $username = $user->user_name;

        $assignedAssetStatus = Cache::remember('assignedAssetStatus', 720, function () {
            return AssetStatus::getStatusID('assigned')->first();
        });

        $description = __('history.AssetCreatedAndAssigned', [
            'assetname'   => $assetData['serial_no'],
            'assetid'     => $assetId,
            'username'    => $username
        ]);

        $assetHistory = [
            'asset_id'            => $assetId,
            'action'              => 'created',
            'new_asset_status_id' => optional($assignedAssetStatus)->id,
            'new_value'           => optional($assignedAssetStatus)->name,
            'new_user_id'         => $discoveryData['user_id'],
            'description'         => $description,
            'created_by'          => $discoveryTool,
            'created_at'          => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'          => Carbon::now()->format('Y-m-d H:i:s'),
        ];

        event(new BulkUpdates($assetHistory));

        $this->updateDiscoveryAssetId($discoveryData, $assetId);
    }

    /**
     * Construct array for creating asset
     */
    public function getAssetData($discoveryData, $technicalSpecs)
    {
        $assetData['asset_type_id']     = $discoveryData['asset_type_id'];
        $assetData['user_id']           = $discoveryData['user_id'];
        $assetData['serial_no']         = $discoveryData['serial_no'];
        $assetData['asset_tag']         = isset($discoveryData['asset_tag']) ? $discoveryData['asset_tag'] : '';
        $assetData['technical_spec_id'] = $technicalSpecs->id;
        $assetData['make_and_model_id'] = $technicalSpecs->make_and_model_id ?? '';
        $assetData['asset_status_id']   = $discoveryData['asset_status_id'];

        $assetData['po_id']             = isset($discoveryData['poNumber']) ? $discoveryData['poNumber'] : '';

        return $assetData;
    }

    /**
     * get the Asset id from serial #
     *
     * @param string $serial
     * @return int
     */
    public function getAssetId($serial)
    {
        if ($serial) {
            $asset = Asset::select('id')->where('serial_no', $serial)->first();

            return $asset ? $asset->id : 0;
        }

        return 0;
    }

    /**
     * get the user id from email address.
     *
     * @param string $email
     *
     * @return int
     */
    public function getUserId($email)
    {
        if ($email) {
            $user = User::select('id')->where('email', $email)->first();

            return  $user ? $user->id : 0;
        }

        return 0;
    }

    /**
     * get the carrier id from carrier.
     *
     * @param string $carrier
     *
     * @return int
     */
    public function getCarrierId($carrier)
    {
        if ($carrier) {
            $carrier = Carrier::select('id')->where('name', $carrier)->first();

            return $carrier ? $carrier->id : null;
        }

        return null;
    }

    /**
     * Auto re assigns the discreapancy assets.
     *
     * @param mixed $assets
     * @param mixed $discoveryTool
     *
     * @return [type]
     */
    public function reAssignDiscrepancyAssets($discoveryAssets, $discoveryTool)
    {
        try {
            $assetStatusId = AssetStatus::where('slug', 'assigned')->first();

            foreach ($discoveryAssets as $discoveryAsset) {
                $asset = Asset::find($discoveryAsset->asset_id);

                if ($asset) {
                    $asset->update([
                        'user_id' => $discoveryAsset->user_id,
                        'asset_status_id' => $assetStatusId->id,
                    ]);
                }

                $description = AssetHistoryService::getAssetHistoryAutoAssignedDescription($discoveryAsset);
                $assetHistory = [
                    'asset_id'    => $discoveryAsset->asset_id,
                    'action'      => 'auto_assigned',
                    'old_user_id' => $discoveryAsset->asset->user_id,
                    'new_user_id' => $discoveryAsset->user_id,
                    'description' => $description,
                    'created_by'  => $discoveryTool,
                    'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
                    'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
                ];

                event(new BulkUpdates($assetHistory));
            }
        } catch (Exception $e) {
            Log::error($e->getMessage());
        }

        return true;
    }
}
