<?php

namespace App\Http\Traits;

use App\Models\AssetStatus;
use Illuminate\Support\Facades\Cache;

trait AssetStatusTrait
{
    public function getUserRequiredAssetStatuses()
    {
        return Cache::remember('bulk-user-asset-status', 7200, function () {
            return AssetStatus::whereIn('slug', ['assigned', 'loaned', 'legal_hold', 'gifted'])->pluck('id')->toArray();
        });
    }

    public function getInstallLocationRequiredAssetStatuses()
    {

        return Cache::remember('bulk-install-asset-status', 7200, function () {
            return AssetStatus::whereIn('slug', ['installed'])->pluck('id')->toArray();
        });
    }

    /**
     * Retrieves the asset statuses required for a storage location from the cache.
     * 
     * @return array The IDs of the required asset statuses.
     */
    public function getStorageLocationRequiredAssetStatuses()
    {
        return Cache::remember('bulk-storage-asset-status', 7200, function () {
            return AssetStatus::whereIn('slug', ['used', 'used_loaner', 'retain_hold', 'legal_hold', 'damaged', 'end_of_life', 'like_new', 'pending_return'])->pluck('id')->toArray();
        });
    }

    public function getStatus(int $assetStatusId)
    {
        return Cache::remember('asset-status-' . $assetStatusId, 7200, function () use ($assetStatusId) {
            return AssetStatus::find($assetStatusId);
        });
    }

    /**
     * Retrieves the required asset statuses for a vendor from the cache.
     *
     * @return array
     */
    public function getVendorRequiredAssetStatuses()
    {
        return Cache::remember('bulk-vendor-asset-status', 7200, function () {
            return AssetStatus::whereIn('slug', ['end_of_life_data_wiped', 'rma_returned'])->pluck('id')->toArray();
        });
    }

    /**
     * Searches for asset statuses that contain the provided status.
     *
     * This function checks the configuration for asset statuses and returns
     * the keys of any statuses that include the given status in their values.
     *
     * @param string $status The status to search for within asset statuses.
     * @return array An array of keys representing asset statuses that contain the provided status.
     */

    public function searchFromStatus($status)
    {
        $array = config('asset-status');
        $return = array();
        foreach ($array as $key => $val) {
            if (in_array($status, $val)) {
                $return[] = $key;
            }
        }
        return $return;
    }
}
