<?php

namespace App\Http\Traits;

use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;

/**
 * This trait is mainly used to bulk asset attribute update
 */
trait AssetAttributeUpdateTrait
{

    /**
     * Get Data for History and Asset
     * @param  array  $inputData
     * @param  int    $count
     * @return array
     */
    public function getAttributeData(array $inputData, $asset = null)
    {
        $dataToupdate['asset'] = [];
        $dataToupdate['history'] = '';
        $dataToupdate = $this->updateValue($dataToupdate, 'asset_type_id', $inputData['asset_type_id'], $asset, 'Asset Type', 'assetType', '\App\Models\AssetType');
        $dataToupdate = $this->updateValue($dataToupdate, 'make_and_model_id', $inputData['make_and_model_id'], $asset, 'Hardware Standard', 'makeAndModel', '\App\Models\MakeAndModel');
        $dataToupdate = $this->updateValue($dataToupdate, 'technical_spec_id', $inputData['technical_spec_id'], $asset, 'Tech Specs', 'technicalSpec', '\App\Models\TechnicalSpecs', 'details');
        $dataToupdate = $this->updateValue($dataToupdate, 'carrier_id', $inputData['carrier_id'], $asset, 'Mobile Carrier', 'carrier', '\App\Models\Carrier');
        $dataToupdate = $this->updateValue($dataToupdate, 'asset_tracking_id', $inputData['asset_tracking_id'], $asset, 'Asset Tracking', 'assetTracking', '\App\Models\AssetTracking');
        $dataToupdate = $this->updateDateTimeValue($dataToupdate, 'created_at', $inputData['created_at'], $asset, 'Created Date');

        // If asset original value does not exist, use technical spec original value
        if ($asset->asset_original_value) {
            $dataToupdate = $this->updateValue($dataToupdate, 'asset_original_value', $inputData['asset_original_value'], $asset, str_replace('(mm/dd/yyyy)', '', 'Asset Original Value'), '', '');
        } else {
            $dataToupdate = $this->updateValue($dataToupdate, 'asset_original_value', $inputData['asset_original_value'], $asset, str_replace('(mm/dd/yyyy)', '', 'Asset Original Value'), 'technicalSpec', '', 'original_value');
        }
        $customAttributes = ['make_and_model_id', 'technical_spec_id', 'carrier_id', 'asset_type_id', 'serial_no', 'created_at', 'asset_tracking_id', 'asset_original_value'];

        //Loop through all assets attributes
        foreach ($this->getHeaderMap() as $key => $attributeName) {
            if (in_array($key, $customAttributes)) {
                continue;
            }
            $dataToupdate = $this->updateValue($dataToupdate, $key, $inputData[$key], $asset, str_replace('(mm/dd/yyyy)', '', $attributeName), '', '');
        }

        return $dataToupdate;
    }

    /**
     * Update the array for update the given Attribute of the asset
     *
     * @param Array $dataToupdate  Existing array of the asset details
     * @param string $key  Attribute name
     * @param string $value  Attribute Value
     * @param Collection $asset  App\Models\Asset
     * @param string $attributeName  Attribute Name
     * @param string $relation Attribute Relation
     * @param string $relationModel Relation Model Name
     * @param string $relationAttribute Relation attribute name
     *
     * @return Array
     */
    public function updateValue($dataToupdate, $key, $value, $asset, $attributeName, $relation = '', $relationModel = '', $relationAttribute = 'name')
    {
        if ($value) {
            $dataToupdate['asset'][$key] = $value;

            if ($relation) {
                $dataToupdate['history'] .= $attributeName . ' updated from ' . optional($asset->$relation)->$relationAttribute . ' to ' . (($relationModel) ? $relationModel::find($value)->$relationAttribute : $value) . ', ';
            } else {
                $dataToupdate['history'] .= $attributeName . ' updated ' . ($asset->$key ? 'from ' . $asset->$key : '') . ' to ' . $value . ', ';
            }
        }

        return $dataToupdate;
    }

    /**
     *
     * Update the array for update the given date Attribute of the asset with datetime
     *
     */
    public function updateDateTimeValue($dataToupdate, $key, $value, $asset, $attributeName)
    {
        if ($value) {
            $dataToupdate['asset'][$key] = ($value != '') ? convert_to_db_datetime($value) : Carbon::now()->format('Y-m-d H:i:s');
            $dataToupdate['history'] .= $attributeName . ' updated ' . ($asset->$key ? 'from ' . $asset->$key : '') . ' to ' . $value . ', ';
        }

        return $dataToupdate;
    }

    /**
     *
     * Update the array for update the given date Attribute of the asset with date
     *
     */
    public function updateDateValue($dataToupdate, $key, $value, $asset, $attributeName)
    {
        if ($value) {
            $dataToupdate['asset'][$key] = ($value != '') ? convert_to_db_date($value) : null;
            $dataToupdate['history'] .= $attributeName . ' updated ' . ($asset->$key ? 'from ' . $asset->$key : '') . ' to ' . $value . ', ';
        }
        return $dataToupdate;
    }
}
