<?php

namespace App\Http\Traits\Asn;

use App\Events\UpdateAssetStatus;
use App\Models\Asset;
use App\Models\AssetHistory;
use App\Models\AssetStatus;
use App\Models\Location;
use App\Models\MakeAndModel;
use App\Models\PurchaseOrder;
use App\Models\TechnicalSpecs;
use Exception;
use Facades\App\Services\Asset\AssetStatusService;
use Facades\App\Services\AssetHistory as RepoAssetHistory;
use Facades\App\Services\JiraDescription as ServicesJiraDescription;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

/**
 * It is used support ASN functionalities like recive asset to a location ..etc
 */
trait AsnAssetTrait
{
    public function recieveAsnAsset($oldStatus)
    {
        $notMappedAssets = [];

        if (!request('location')) {
            return response()->json('Location field is empty!');
        }

        if (!request('ticket')) {
            return response()->json('Ticket field is empty!');
        }

        $newStatus = $this->getAsnRecieveAssetStatus();

        foreach (request('received') as $assetId) {
            try {
                $asset = Asset::where('asset_status_id', $oldStatus->id)->findOrFail($assetId);

                if (!$this->validateAssetFields($asset)) {
                    array_push($notMappedAssets, $asset->serial_no);
                    continue;
                }

                $assetData      = $this->getDataToReceive($newStatus);
                $assetHistory   = $this->getHistoryDataToReceive($asset, $oldStatus, $newStatus);

                $statusName = $newStatus->name == "Assigned" ? "AsnAssigned" : "AsnRecieved";

                $jiraDescription = ServicesJiraDescription::getStatusDescription($statusName, $assetHistory, $asset, request('comments'));

                event(new UpdateAssetStatus($jiraDescription, request('ticket'), $assetHistory, Auth::user()->id));

                $asset->update($assetData);
            } catch (Exception $e) {
                Log::error($e->getMessage());
            }
        }

        if (!empty($notMappedAssets)) {
            return response()->json(implode(', ', $notMappedAssets) . ' - These Asset(s) were not mapped, Please Map them before receiving.');
        }

        return response()->json('Success');
    }

    /**
     * Validate to check the asset fields in unmapped values
     * @param mixed $asset
     *
     * @return bool
     */
    public function validateAssetFields($asset)
    {
        $notMappedValues = [null, '', 0];
        if (in_array($asset->asset_type_id, $notMappedValues)) {
            return false;
        }

        if (in_array($asset->make_and_model_id, $notMappedValues)) {
            return false;
        }

        if (in_array($asset->technical_spec_id, $notMappedValues)) {
            return false;
        }
        if ($asset->po_id == '' || $asset->po_id == null) {
            return false;
        }

        return true;
    }

    /**
     * Creating array for updating asset as Received
     *
     * @param AssetStatus $status
     * @return Array
     */
    public function getDataToReceive($status)
    {
        $assetData['location_id']       = request('location');
        $assetData['asset_status_id']   = $status->id;
        $assetData['user_id']     = null;
        $assetData['ticket_no']         = request('ticket');

        return $assetData;
    }

    /**
     * Creating array for update the asset history for Received
     *
     * @param Asset $asset
     * @return Array
     */
    public function getHistoryDataToReceive($asset, $oldStatus, $newStatus)
    {
        $description = __('history.Created', [
            'assetname'     => $asset->serial_no,
            'assetid'       => $asset->id,
            'newroomname'   => optional(Location::find(request('location')))->room_name,
        ]);

        $assetHistoryData = [
            'user_id'           => Auth::id(),
            'asset_id'          => $asset->id,
            'ticket_no'         => request('ticket'),
            'new_location_id'   => request('location'),
            'old_asset_status_id' => $oldStatus->id,
            'new_asset_status_id' => $newStatus->id,
            'old_value'   => $oldStatus->name,
            'new_value'   => $newStatus->name,
            'action'            => 'asset_received',
            'description'       => $description,
        ];

        return $assetHistoryData;
    }


    /**
     * Mark Single Item Received
     *
     * @return void
     */
    public function markSingleItemReceived()
    {
        if ((!empty(request('location_id'))) && (!empty(request('user_id')))) {
            return response()->json(['status' => "Error", 'message' => "Need to select a valid Location or User"]);
        }
        $asset = Asset::findOrFail(request('asset_id'));

        if (!$this->validateAssetFields($asset)) {
            return response()->json(['status' => "Error", 'message' => $asset->serial_no . " - This Asset were not mapped. Please map them before receiving."]);
        }

        $updateData = $this->singleAssetReceiveData($asset);

        $this->createUpdateHistory($updateData['old_data'], $updateData['status'], request()->all(), $asset);

        $asset->update($updateData['update']);

        return response()->json(['status' => "Success", 'message' => $updateData['message']]);
    }

    /**
     * single Asset Recive Data array
     * @return [type]
     */
    public function singleAssetReceiveData($asset)
    {

        $update = [];
        $ticketNo = (request('ticket_no') != null) ? request('ticket_no') : ($asset ? $asset->ticket_no : null);
        $update['ticket_no'] = $ticketNo;

        if (!empty(request('location_id'))) {
            $status = $this->getAsnRecieveAssetStatus();
            $update['location_id']     = request('location_id');
            $update['user_id'] = null;
            $update['asset_status_id'] = $status->id;
            $message = 'Asset ' . $asset->serial_no . ' recieved to the selected location.';
        } else {
            $status = $this->getAsnAssignAssetStatus();
            $update['user_id']     = request('user_id');
            $update['location_id'] = null;
            $update['asset_status_id'] = $status->id;
            $message = 'Asset ' . $asset->serial_no . ' Assigned to selected user.';
        }

        $oldData['location_id'] = $asset->location_id;
        $oldData['user_id'] = $asset->user_id;
        $oldData['asset_id'] = $asset->id;
        $oldData['ticket_no'] = $ticketNo;

        return ['update' => $update, 'message' => $message, 'status' => $status, 'old_data' => $oldData];
    }

    /**
     * Create Update History
     *
     * @param  Eloquent $asset
     * @param  Eloquent $status
     * @param  array    $inputData
     *
     * @return void
     */
    public function createUpdateHistory($oldData, $status, $inputData, $asset)
    {
        $comments = '';
        $description = RepoAssetHistory::getAsnAssetReceivedDescription($status, $asset, $comments, $inputData);

        $assetHistory = [
            'user_id'         => Auth::id(),
            'asset_id'        => $oldData['asset_id'],
            'ticket_no'       => $oldData['ticket_no'],
            'old_location_id' => $oldData['location_id'] ? $oldData['location_id'] : NULL,
            'new_location_id' => $inputData['location_id'] ? $inputData['location_id'] : NULL,
            'old_user_id'     => $oldData['user_id'] ? $oldData['user_id'] : NULL,
            'new_user_id'     => $inputData['user_id'] ? $inputData['user_id'] : NULL,
            'old_asset_status_id' => $asset->asset_status_id,
            'new_asset_status_id' => $status->id,
            'old_value'       => $asset->assetStatus->name,
            'new_value'       => $status->name,
            'action'          => 'asset_received',
            'description'     => $description,
        ];


        if ($status->name == "Assigned") {
            $statusName = "AsnAssigned";
        } else {
            $statusName = "AsnRecieved";
        }

        $jiraDescription = ServicesJiraDescription::getStatusDescription($statusName, $assetHistory, $asset, request('comments'));

        event(new UpdateAssetStatus($jiraDescription, $oldData['ticket_no'], $assetHistory, Auth::user()->id));
    }

    /**
     * Makes the array of data for update
     * @return [type]
     */
    public function getDataToUpdate()
    {
        return [
            'asset_tag'         => request('asset_tag'),
            'serial_no'         => request('serial_no'),
            'make_and_model_id' => request('make_and_model_id'),
            'po_id'         => request('po_id'),
            'asset_type_id'     => request('asset_type_id'),
            'technical_spec_id'     => request('technical_spec'),
        ];
    }


    /**
     * Get Asset Details
     *
     * @param int $id
     *
     * @return Array
     */
    public function getAsset($id)
    {
        $asset = Asset::find($id);

        $data['asset_tag']   = $asset->asset_tag;
        $data['serial_no']   = $asset->serial_no;
        $data['po']          = $asset->po_id;
        $data['hardware']    = $asset->make_and_model_id;
        $data['techspec']    = $asset->technical_spec_id;
        $data['description'] = $asset->part_no;
        $data['assettype']   = $asset->asset_type_id;
        $data['part_no']     = $asset->part_no;

        return $data;
    }

    /**
     * Get Hardware standard name and id
     *
     * @return Mixed
     */
    public function getHardware()
    {
        $hardwareStandards = MakeAndModel::where('asset_type_id', '=', request('assettype'))->get()->pluck('name', 'id');

        return $hardwareStandards ?? '';
    }

    /**
     * Get Technical specs details and id
     *
     * @return Mixed
     */
    public function getTechSpech()
    {
        $techSpecs = TechnicalSpecs::where('make_and_model_id', '=', request('techSpec'))->get()->pluck('details', 'id');

        return $techSpecs ?? '';
    }

    /**
     * The asset status for recieving the asset from ASN
     * @return collection
     */
    public function getAsnRecieveAssetStatus()
    {
        return AssetStatus::where('slug', AssetStatusService::whenCreatingAsset())->first();
    }

    /**
     * Get the asset status for assigning the asset from ASN
     *
     * @param array $params The query parameters
     * @return \Illuminate\Database\Eloquent\Collection The asset status for assigning the asset from ASN
     */
    public function getAsnAssignAssetStatus($params = [])
    {
        $query = AssetStatus::query();

        // If 'select' key is provided and is an array, use it for selecting specific columns
        if (!empty($params['select']) && is_array($params['select'])) {
            $query->select($params['select']);
        }

        return $query->where('slug', 'assigned')->first();
    }


    /**
     * update assets with brand_new status
     * @param array $received list of asset ids
     * @param string $status status slug
     */
    public function receiveAsnAsset($received, $status)
    {
        $notMappedAssets = [];
        $oldStatus = AssetStatus::where('slug', $status)->first();
        $newStatus = $this->getBrandNewstatus();

        foreach ($received as $assetId) {
         
            try {

                $asset = Asset::findOrFail($assetId);
              
                if (!$this->validateAssetFields($asset)) {

                    array_push($notMappedAssets, $asset->serial_no);
                    continue;
                }

                $assetData = $this->getDataToReceive($newStatus);
              
                if (optional(optional($asset->makeAndModel)->manufacturer)->slug == 'apple' && starts_with(lcfirst($asset->serial_no), 's')) {
                  
                    $assetData['serial_no'] = ltrim(lcfirst($asset->serial_no), 's');
                    if ($this->updateExistingAssetAndRemoveDuplicate($asset, $assetData['serial_no'])) {
                        continue;
                    }
                }

                $assetHistory = $this->getHistoryDataToReceive($asset, $oldStatus, $newStatus);
                $asset->update($assetData);
                AssetHistory::create($assetHistory);

            } catch (Exception $e) {
              
                Log::channel('daily')->info("failed to receive ASN asset#$asset->serial_no: " . $e->getMessage());
            }
        }

        return $notMappedAssets;
    }

    /**
     * Delete unwanted parts from inventory
     *
     * @param array $allMapping the unwanted mapping data
     * @param int $statusId the status id of the assets to be deleted
     *
     * @return void
     */
    public function removeUnWantedParts($allMapping, $statusId)
    {
        // Ensure we have a collection (works with both arrays and collections)
        $partNumbers = collect($allMapping)->pluck('part_no')->toArray();

        DB::transaction(function () use ($partNumbers, $statusId) {
          
            // Get asset IDs to delete
            $assetIds = Asset::where('asset_status_id', $statusId)
                ->where(function ($query) use ($partNumbers) {
                    $query->whereIn('description', $partNumbers)
                        ->orWhereIn('part_no', $partNumbers);
                })
                ->pluck('id');

            if ($assetIds->isNotEmpty()) {
            
                // Bulk delete related histories
                AssetHistory::whereIn('asset_id', $assetIds)->delete();

                // Bulk delete assets
                Asset::whereIn('id', $assetIds)->delete();
            }
        });
    }

    /**
     * Update Single Asset
     * 
     * @param Eloquent $asset
     * @param  array   $inputData
     *
     * @return string
     */
    public function updateSingleAsset($asset, $inputData)
    {
        $ticketNo = (request('ticket_no') != null) ? request('ticket_no') : ($asset ? $asset->ticket_no : null);

        $update = [];
        $update['ticket_no'] = $ticketNo;
        $update['serial_no'] = $asset->serial_no;

        if (optional(optional($asset->makeAndModel)->manufacturer)->slug == 'apple' && starts_with(lcfirst($asset->serial_no), 's')) {
          
            $update['serial_no'] = ltrim(lcfirst($asset->serial_no), 's');
           
            if ($this->updateExistingAssetAndRemoveDuplicate($asset, $update['serial_no'])) {
                
                return 'Asset ' . $update['serial_no'] . ' updated successfully.';
            }
        }

        if (!empty($inputData['location_id'])) {
          
            $status = $this->getBrandNewstatus();
            $update['location_id']      = $inputData['location_id'];
            $update['asset_status_id']  = $status->id;

            $message = 'Asset ' . $update['serial_no'] . ' changed to Brand New status, and moved to selected location.';
        } else {
        
            $status = $this->getAsnAssignAssetStatus(['select' => ['id']]);
            $update['user_id']          = $inputData['user_id'];
            $update['location_id']      = null;
            $update['asset_status_id']  = $status->id;
            $message = 'Asset ' . $update['serial_no'] . ' Assigned to selected user.';
        }

        $oldData['location_id'] = $asset->location_id;
        $oldData['user_id'] = $asset->user_id;
        $oldData['asset_id'] = $asset->id;
        $oldData['ticket_no'] = $ticketNo;

        $this->createUpdateHistory($oldData, $status, $inputData, $asset);

        $asset->update($update);

        return $message;
    }

    /**
     * Retrieve and cache the 'brand_new' asset status.
     *
     * This function attempts to retrieve the asset status with the slug 'brand_new'
     * from the database and caches the result for 1 hour (3600 seconds).
     *
     * @return AssetStatus|null The 'brand_new' AssetStatus model or null if not found.
     */
    public function getBrandNewstatus()
    {
        return Cache::remember('brand_new_status', 3600, function () {
            return AssetStatus::where('slug', 'brand_new')->first();
        });
    }

      /**
     * Get Asset Details
     *
     * @param int $id
     *
     * @return Array
     */
    public function getAssetDetails($id)
    {
        $asset = Asset::find($id);

        $data['serial_no']   = $asset->serial_no;
        $data['po']          = $asset->po_id ?? '';
        $data['hardware']    = $asset->make_and_model_id;
        $data['techspec']    = $asset->technical_spec_id;
        $data['description'] = $asset->description;
        $data['assettype']   = $asset->asset_type_id;

        return $data;
    }

        /**
     * update purchase order details of assets
     * @param array $assets list of asset ids
     * 
     * @return string
     */
    public function bulkUpdateAssetPoDetails($assets)
    {
        foreach ($assets as $key => $assetId) {
           
            if (request('po_id')) {
               
                $po = PurchaseOrder::find(request('po_id'));
                $poId = $po ? $po->number : null;
            }
            Asset::findOrFail($assetId)->update(['po_id' => $poId ?? null,]);
        }

        return 'Success';
    }

    /**
     * Updates an existing asset if given serial# exists and removes asn asset.
     *
     * @param mixed $asset The asset to be updated.
     * @param mixed $serialNo The serial number of the asset.
     * @return bool Returns true if the update and removal was done, false otherwise.
     */
    public function updateExistingAssetAndRemoveDuplicate($asset, $serialNo)
    {
        $existing = Asset::where('serial_no', $serialNo)->where('id', '!=', $asset->id)->first();
       
        if (!$existing) {
            return false;
        }

        $existing->update(['asset_original_value' => $asset->asset_original_value]);
        AssetHistory::where('asset_id', $asset->id)->delete();
        $asset->delete();

        return true;
    }
}
