<?php

namespace App\Http\Requests\SoftwareLicense;

use App\Models\SoftwareLicense\SoftwareLicense;
use App\Rules\SoftwareLicense\LicenseKeyFormat;
use App\Rules\SoftwareLicense\LicenseKeysCount;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class LicenseRequest extends FormRequest
{

    public function authorize()
    {
        return true;
    }


    public function rules()
    {
        if ($this->has('id')) {
            return [
                'software_license_category_id' => 'required',
                'software_license_manufacturer_id' => 'required',
                'license_type' => 'required',
                'name' => 'required',
                'type' => 'required',
                'license_key_type' => ['required', Rule::in(array_keys(config('software-license.license_key_types')))],
                'key_usage_type' => [Rule::requiredIf(function () {
                    return in_array(request('license_key_type'), ['single_license', 'user_license']);
                }), request('key_usage_type') ? Rule::in(array_keys(config('software-license.key_usage_types'))) : ''],
                'users_allowed_for_key' => 'nullable',
                'licenses_purchased'  => ['nullable', 'integer', Rule::requiredIf(function () {
                    return in_array(request('license_key_type'), ['user_license']);
                })],
                'subscription_expiration' => $this->license_type == 'Subscription' ? 'nullable|date|after_or_equal:start_date' : '',
                'software_liceneses_documents.*' => 'file|mimes:pdf|max:5120',
            ];
        }

        return [
            'software_license_category_id' => 'required',
            'software_license_manufacturer_id' => 'required',
            'name' => 'required',
            'type' => 'required',
            'license_type' => 'required',
            'billing_cycle' => 'required',
            'start_date' => 'required',
            'payment_method_id' => 'required',
            'status' => 'required',
            'license_key_type' => ['required', Rule::in(array_keys(config('software-license.license_key_types')))],
            'key_usage_type' => [Rule::requiredIf(function () {
                return in_array(request('license_key_type'), ['single_license', 'user_license']);
            }), request('key_usage_type') ? Rule::in(array_keys(config('software-license.key_usage_types'))) : ''],
            'license_key' => 'required_if:license_key_type,=,single_license',
            'software_license_subscription_id' => 'required_if:license_type,=,subscription',
            'subscription_expiration' => $this->license_type == 'Subscription' ? 'nullable|date|after_or_equal:start_date' : '',
            // 'license-key-choice' => 'required_if:license_key_type,=,user_license',
            'software_liceneses_keys' => 'required_if:license-key-choice,=,license_key_upload',
            'license_keys' => 'required_if:license-key-choice,=,license_key_text_area',
            'users_allowed_for_key' => [Rule::requiredIf(function () {
                return (request('license_key_type') != 'no_license' && request('key_usage_type') == 'limited_usage');
            })],
            'license_key' => [new LicenseKeyFormat('/^[a-zA-Z0-9,_ -]+$/i')],
            'license_keys' => [new LicenseKeyFormat('/^[a-zA-Z0-9,_\r\n -]+$/i'), new LicenseKeysCount(request('licenses_purchased'))],
            'licenses_purchased'  => ['nullable', 'integer', Rule::requiredIf(function () {
                return in_array(request('license_key_type'), ['user_license']);
            })],
            'software_liceneses_documents.*' => 'file|mimes:pdf|max:5120',
        ];
    }

    public function messages()
    {
        return [
            'software_license_category_id.required' => 'Please select a license category',
            'software_license_manufacturer_id.required' => 'Please select a license manufacturer',
            'name.required' => 'Please enter the name',
            'type.required' => 'Please select a type.',
            'license_type.required' => 'Please select a license type',
            'billing_cycle.unique' => 'Please select a billing cycle',
            'start_date.required' => 'Please enter the contract start date',
            'payment_method_id.required' => 'Please select a payment method',
            'status.required' => 'Please select a status',
            'license_key_type.in' => 'Invalid Software license type',
            'license_key.required' => 'Please enter license key',
            'license_keys.required' => 'Please enter license keys line by line.',
            'license-key-choice.required' => 'Please select how to add keys',
            'software_liceneses_keys.required' => 'Please select a file to upload',
            'key_usage_type.required' => 'Please select key usage type',
            'key_usage_type.in' => 'Invalid key usage type',
            'users_allowed_for_key.required' => 'Please enter the # of users allowed',
            'licenses_purchased.required' => 'Please enter the # of licenses purchased',
            'software_license_subscription_id.required' => 'Please select subscription length',
            'software_liceneses_documents.*.file' => 'Each uploaded document must be a file.',
            'software_liceneses_documents.*.mimes' => 'Each document must be a PDF.',
            'software_liceneses_documents.*.max' => 'Each document must not be greater than 5MB.',
        ];
    }
}
