<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;

class CsvFileUploadRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return true; // Allow all users for now
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        return [
            'file' => [
                'required',
                'file',
                'mimes:csv,xls,xlsx,txt',
                'max:5000',
            ],
        ];
    }

    /**
     * Get the custom error messages for the validator.
     *
     * @return array
     */
    public function messages()
    {
        return [
            'file.required' => 'Please upload a file.',
            'file.mimes' => 'Only CSV, Excel (.xls, .xlsx) files are allowed.',
            'file.max' => 'The file size must not exceed 5MB.',
        ];
    }

    /**
     * Handle additional validations for content-type and file MIME type.
     *
     * @return \Illuminate\Http\Response|null
     */
    public function withValidator($validator)
    {
        $validator->after(function ($validator) {
            $file = $this->file('file');
            $mimeType = $file->getMimeType();

            // Define allowed MIME types for CSV and Excel files.
            $allowedMimeTypes = [
                'text/csv', // MIME type for CSV files.
                'text/plain', // MIME type for .csv files.
                'application/vnd.ms-excel', // MIME type for .xls files.
                'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet', // MIME type for .xlsx files.
            ];

            // Check if the MIME type matches the allowed types.
            if (!in_array($mimeType, $allowedMimeTypes)) {
                $validator->errors()->add('file', 'Invalid Content-Type. Only CSV and Excel files are allowed.');
            }

            // Define allowed file extensions.
            $allowedExtensions = ['csv', 'xls', 'xlsx'];

            // Check the file extension.
            $extension = strtolower($file->getClientOriginalExtension());
            if (!in_array($extension, $allowedExtensions)) {
                $validator->errors()->add('file', 'Invalid file extension. Only .csv, .xls and .xlsx files are allowed.');
            }
        });
    }

    /**
     * Format the validation error response before returning it.
     *
     * @param  \Illuminate\Contracts\Validation\Validator  $validator
     * @return void
     */
    protected function failedValidation(Validator $validator)
    {
        $errors = $validator->errors()->getMessages();
        $response = [
            'errors' => view('assets.partials.upload-validation-errors', compact('errors'))->render(),
        ];

        throw new HttpResponseException(
            response()->json($response, 422)
        );
    }
}
