<?php

namespace App\Http\Controllers\ToolBox;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\ApiClient;
use App\Models\OAuthScope;
use App\Models\SystemSetting;
use App\Repositories\ClientRepository;
use App\Services\GitBookJwtService;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Laravel\Passport\Token;


/**
 * Controller class for managing API Users in the Toolbox.
 */
class ApiUserController extends Controller
{
    /**
     * The client repository instance.
     *
     * @var \App\Repositories\ClientRepository
     */
    protected $clientRepository;
    protected $jwtService;

    /**
     * Create a new ApiUserController instance.
     *
     * @param ClientRepository $clientRepository
     */
    public function __construct(ClientRepository $clientRepository, GitBookJwtService $jwtService)
    {
        $this->clientRepository = $clientRepository;
        $this->jwtService = $jwtService;
    }

    /**
     * Display a listing of the API clients.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $jwtToken = $this->jwtService->generateJwtForGitBook('api');
        $clients = ApiClient::with('addedBy')->with('tokens')->whereRevoked(0)->orderBy('name')->get();

        return view('toolbox.api-users', compact('clients', 'jwtToken'));
    }

    /**
     * Store a newly created API client.
     *
     * @param Request $request
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $this->validate(request(), [
            'name' => [
                'required',
                Rule::unique('oauth_clients')->where(function ($query) {
                    $query->where('revoked', '<>', 1);
                })
            ]
        ]);

        $name = $request->name;
        $notes = $request->notes;
        $validity = $request->validity;

        $client = $this->clientRepository->create(
            null,
            $name,
            '',
            $notes,
            $validity
        );

        if ($request->has('permissions')) {

            $permissions = $request->permissions;

            $client = $this->clientRepository->find($client->id);

            $client->oAuthscopes()->sync($permissions);
        }

        $token = $this->generateToken($client->fresh());

        $validity = explode(' ', $validity);

        $client->tokens()->whereRevoked(false)->update(['expires_at' => now()->add($validity[0], $validity[1])]);

        // $content = "Teqtivity API Credentials \n";
        // $content .= "Client Id: " . $client->uuid . " \n";
        // $content .= "Client Name: " . $client->name . " \n";
        // $content .= "Client Secret: " . $client->secret . " \n";
        // $content .= "Access Token: " . $token . " \n";

        return redirect(route('api-users.index'))
            ->with('message', __('message.user_created'))
            ->with('client', $client)
            // ->with('content', $content)
            ->with('token', $token)
            ->with('status', true);
    }

    /**
     * Update an existing API client.
     *
     * @param Request $request
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request)
    {
        $this->validate(request(), [
            'name' => [
                'required',
                Rule::unique('oauth_clients')->where(function ($query) use ($request) {
                    $query->where('revoked', '<>', 1)->where('id', '<>', $request->id);
                })
            ]
        ]);

        $client = $this->clientRepository->find(request('id'));

        $this->clientRepository->update(
            $client,
            request('name'),
            '',
            request('notes')
        );

        return response()->json('success');
    }

    /**
     * Destroy an API client.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy()
    {
        $client = $this->clientRepository->find(request('id'));

        $this->clientRepository->delete($client);

        return response()->json(__('message.deleted'));
    }

    /**
     * Handle Ajax requests for editing and deleting API clients.
     *
     * @param Request $request
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function ajax(Request $request)
    {
        if (request('action') == 'edit') {
            return $this->update($request);
        } elseif (request('action') == 'delete') {
            return $this->destroy();
        }

        return response()->json(true);
    }

    /**
     * Refresh an API client's token.
     *
     * @param string $id
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function refreshToken($id)
    {
        $client = $this->clientRepository->find($id);
        $client->tokens()->update(['revoked' => true]);

        $apiClient = $this->clientRepository->find($id);

        $token = $this->generateToken($apiClient);

        return redirect(route('api-users.index'))
            ->with('message', 'Token Refreshed successfully')
            ->with('client', $client)
            ->with('token', $token)
            ->with('status', true);
    }

    /**
     * Generate an access token for the API client.
     *
     * @param ApiClient $client
     *
     * @return string
     */
    public function generateToken($client)
    {
        $scopes = "";

        if ($client->oAuthscopes) {
            $scopes = implode(' ', $client->oAuthscopes->pluck('slug')->toArray());
        }

        $response = (new \GuzzleHttp\Client)->post(url('oauth/token'), [
            'form_params' => [
                'grant_type' => 'client_credentials',
                'client_id' => $client->id, // Client ID
                'client_secret' => $client->secret, // Client secret
                'scope' => $scopes
            ]
        ]);

        return json_decode((string) $response->getBody(), true)['access_token'];
    }

    /**
     * Set the scopes for a given API client's token.
     *
     * @param  \Illuminate\Http\Request  $request
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function setTokenScopes(Request $request)
    {
        $clientId = $request->client_id;
        $permissions = $request->permissions;

        $apiClient = $this->clientRepository->find($clientId);

        if ($apiClient) {

            if ($permissions) {
                $newScopes = OAuthScope::whereIn('id', $permissions)->pluck('slug')->toArray();
                $apiClient->oAuthscopes()->sync($permissions);
            } else {
                $newScopes = [];

                $apiClient->oAuthscopes()->detach();
            }

            $token = Token::where('client_id', $apiClient->id)
                ->whereRevoked(0)
                ->first();

            if ($token) {
                $token->scopes = $newScopes;
                $token->save();
            }
        }

        return redirect(route('api-users.index'))
            ->with('message', __('message.scopes_updated'));
    }

    public function createApiScopes(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|unique:oauth_scopes',
            'description' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 200);
        }

        $permissionData = $request->only(['name', 'description']);

        $permissionData['slug'] = Str::slug($permissionData['name']);

        OAuthScope::create($permissionData);

        session()->flash('message', 'API Permission created successfully.');

        return response()->json(['success' => 'API Permission created successfully!'], 200);
    }

    /**
     * Handle the callback from GitBook authentication
     *
     * @return object
     */
    public function gitBookAuthCallback()
    {
        $locationToRedirect = request('location');
        $jwtToken = $this->jwtService->generateJwtForGitBook('api');
        $systemSetting = optional(SystemSetting::where('slug', 'api_guide_url')->first())->value;

        return redirect($systemSetting . $locationToRedirect . '/?jwt=' . $jwtToken);
    }
}
