<?php

namespace App\Http\Controllers\Terminations\TermsReport;

use App\Http\Controllers\Controller;
use App\Http\Responses\DataTableJsonResponse;
use App\Http\Traits\SavedSearchTrait;
use App\Models\TermAssetsComment;
use Carbon\Carbon;
use App\Services\Terminations\TermsReport\TermsReportData;
use App\Services\Terminations\TermsReport\TermsReportExport;
use Illuminate\Http\Request;
use App\Models\Location;
use App\Models\SavedSearch;
use App\User;
use Cache;
use Exception;
use Illuminate\Support\Facades\Auth;

class TermsReportController extends Controller
{
    use SavedSearchTrait;

    /**
     * Initialize the controller with the TermsReportData and TermsReportExport services.
     *
     * @param \App\Services\TermsReportData $termsReportData The service to manage terms report data
     * @param \App\Services\TermsReportExport $termsReportExport The service to manage terms report export
     */
    public function __construct(protected TermsReportData $termsReportData, protected TermsReportExport $termsReportExport)
    {
    }

    /**
     * Display the index view for the terms report.
     *
     * @return \Illuminate\View\View The view for the terms report index
     */
    public function index()
    {
        $userId         = Auth::id();
        $user           = User::with('userType')->find($userId);
        $savedFilters = SavedSearch::reportName('terms_report')->userSavedSearch($userId)->get();
        $viewAllFilters = SavedSearch::reportName('terms_report')->adminSavedSearch()->get();
        $defaultStatuses = ['loaned', 'assigned'];
        $savedSearch    = [];
        $search         = [];

        if (request('admin_saved_search') || request('user_saved_search')) {
            try {
                if (request('admin_saved_search')) {
                    $savedSearch = SavedSearch::reportName('terms_report')->where('id', request('admin_saved_search'))->where('view_all', 1)->firstOrFail();
                }

                if (request('user_saved_search')) {
                    $savedSearch = SavedSearch::reportName('terms_report')->where('id', request('user_saved_search'))->where('user_id', Auth::id())->whereNull('view_all')->firstOrFail();
                }

                $search = json_decode($savedSearch->parameters, true);
            } catch (Exception $e) {
                return redirect(route('terms-report'))->with('error', 'Search Not Found!');
            }
        }

        if (request('fte')) {
            $search['fte'] = request('fte');
        }

        if (request('from')) {
            $search['date_from'] = request('from');
        }

        if (request('to')) {
            $search['date_to'] = request('to');
        }

        return view('terminations.terms-report.terms-report-index', compact('savedFilters', 'viewAllFilters', 'savedSearch', 'search', 'user', 'defaultStatuses'));
    }

    /**
     * Retrieve and format the data for the terms report.
     *
     * @param \Illuminate\Http\Request $request The incoming request instance
     *
     * @return \Illuminate\Http\JsonResponse The JSON response containing the formatted report data
     */
    public function data(Request $request)
    {
        $filteredData   = $this->termsReportData->filter();
        $users          = $filteredData['users'];
        $totalData      = $filteredData['count'];

        $start  = request('start');
        $data   = [];

        if (!empty($users)) {
            $data = $this->termsReportData->getUserData($users, $start, $data);
        }

        return new DataTableJsonResponse($request->input('draw'), $data, $totalData);
    }

    /**
     * Export the terms report data to a CSV file.
     *
     * @return \Symfony\Component\HttpFoundation\StreamedResponse The response to export the data as CSV
     */
    public function export()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $data = [];
        $datas = collect();

        $filteredData = $this->termsReportExport->filterExportData()->get();
        $datas->push($this->termsReportExport->getExportUserData($filteredData, 0, $data));

        return exportToCsv($datas->toArray());
    }

    /**
     * View the term assets for a specific user.
     *
     * @param int $userId The ID of the user
     *
     * @return mixed The view data for the uncollected assets
     */
    public function viewTermAsset($userId)
    {
        $termAssets = $this->termsReportData->getAllUncollectedUserAssets($userId);

        return $this->termsReportData->getUncollectedAssetViewData($termAssets);
    }

    /**
     * Update the comment for a term asset.
     *
     * @param int $id The ID of the user
     *
     * @return \Illuminate\Http\RedirectResponse The redirect response to the terms report
     */
    public function updateComment($id)
    {
        $data = [
            'comment' => request('comment'),
            'user_id' => $id,
            'added_by' => Auth::id(),
        ];

        TermAssetsComment::create($data);

        return redirect('terms-report');
    }

    /**
     * Retrieve and format comments for a specific user's term assets.
     *
     * @param int $id The ID of the user
     *
     * @return string The formatted HTML string containing the comments
     */
    public function getComment($id)
    {
        $termAssets = TermAssetsComment::with('user', 'addedBy')->where('user_id', $id)->orderBy('created_at', 'DESC')->get();

        $commentString = '<ul>';

        foreach ($termAssets as $termAsset) {
            $commentString .= '<li>' . $termAsset->comment . ' By <b>' . $termAsset->addedBy->email . '</b> at ' . $termAsset->created_at->diffForHumans() . '</li>';
        }

        $commentString .= '</ul>';

        return $commentString;
    }

    /**
     * Retrieve saved search parameters from the request.
     *
     * @return array The array containing the saved search parameters
     */
    public function getSavedSearchParameters()
    {
        return [
            'no_of_assets'        => request('no_of_assets') != '' ? request('no_of_assets') : '',
            'work_location'       => request('work_location') ? request('work_location') : '',
            'employee_type'       => request('employee_type') ? request('employee_type') : '',
            'employee_name'       => request('employee_name') ? request('employee_name') : '',
            'date_from'           => request('date_from') ? request('date_from') : '',
            'date_to'             => request('date_to') ? request('date_to') : '',
            'status'              => request('status') ? request('status') : '',
            'has_ticket'          => request('has_ticket') ? request('has_ticket') : '',
            'legal_hold'          => request('legal_hold') ? request('legal_hold') : '',
            'ticket_no'           => request('ticket_no') ? request('ticket_no') : '',
            'outstanding_days'    => request('outstanding_days') ? request('outstanding_days') : '',
            'fte'                 => request('fte') ? request('fte') : '',
        ];
    }

    /**
     * Retrieve the report name for saved searches.
     *
     * @return string The report name for saved searches
     */
    public function getSavedSearchReportName()
    {
        return 'terms_report';
    }
}
