<?php

namespace App\Http\Controllers\Terminations\Retriever;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Http\Responses\ReportJsonResponse;
use App\Services\Terminations\Retriever\RetrieverReportService;
use App\Services\Terminations\Retriever\RetrieverShipmentService;

/**
 * RetrieverReportController
 */
class RetrieverReportController extends Controller
{

    /**
     * 
     */
    public function __construct(protected RetrieverReportService $service, protected RetrieverShipmentService $shipmentService)
    {
    }


    /**
     * Retrieves the index view for the RetrieverReportController.
     *
     * This function checks the connection status of the RetrieverShipmentService and
     * passes it to the 'terminations.retriever.index' view.
     *
     * @return \Illuminate\Contracts\View\View The index view.
     */
    public function index()
    {
        $connectionStatus = $this->shipmentService->checkConnection();
        return view('terminations.retriever.index', compact('connectionStatus'));
    }

    /**
     * data
     *
     * @param  mixed $request
     * @return json
     */
    public function data(Request $request)
    {
        setUnlimitedExecutionTimeAndMemoryLimit();
        
        $data           = [];
        $filteredData   = $this->service->data(1);
        $assets         = $filteredData['assets'];
        $start          = $request->input('start');

        if (!empty($assets)) {
            $data = $this->service->getReportData($assets, $start, $data, 1);
        }
        return new ReportJsonResponse($request->input('draw'), $data, $filteredData['count']);
    }

    /**
     * export
     *
     * @return void
     */
    public function export()
    {
        setUnlimitedExecutionTimeAndMemoryLimit();

        $datas = collect();
        $data = [];
        $filteredData = $this->service->data();
        $filteredData->chunk(5000, function ($dataChunks) use ($data, $datas) {
            $datas->push($this->service->getReportData($dataChunks, 0, $data));
        });
        return exportToCsv($datas->toArray());
    }

    /**
     * Retrieves the user assets from the Teqtivity service and returns a view with the assets.
     *
     * @param int $userId The ID of the user.
     * @return \Illuminate\Contracts\View\View The view with the user assets.
     */
    public function getUserAssets($userId)
    {
        $userTeqAssets = $this->service->getUserTeqAssets($userId);

        return view('terminations.retriever.partials.user-teq-assets', compact('userTeqAssets'));
    }

    /**
     * Autofil for the filter fields
     */
    public function getFilterFields(Request $request)
    {
        if (!empty($request->search)) {

            $result = $this->service->getFilterFields($request->search, $request->type);

            if (!$result->count()) {
                $result[] = ['id' => '0', 'label' => 'No result found', 'data' => ''];
            }

            return response()->json($result);
        }
    }

    /**
     * retrieves the deactivated reason for a given retriever ID.
     * 
     * @param retriverId 
     * 
     * @return the deactivated reason for a given retriever ID.
     */
    public function getDeactivatedReason($retriverId)
    {
        $deactivatedReason = $this->service->getDeactivatedReason($retriverId);

        return $deactivatedReason;
    }
}
